// GetCamera.java

import java.awt.Color;				// import colour
import java.awt.Font;				// import font
import java.awt.Graphics2D;			// import 2D graphics
import java.awt.image.BufferedImage;		// import buffered image
import java.awt.image.WritableRaster;		// import writeable raster

import java.io.File;				// import file
import java.io.IOException;			// import I/O exception

import java.text.SimpleDateFormat;		// import simple date format

import java.util.Base64;			// import base64
import java.util.Base64.Encoder;		// import base64 encoder
import java.util.Date;				// import date
import java.util.Iterator;			// import iterator

import javax.imageio.ImageIO;			// import image I/O
import javax.imageio.IIOImage;			// import image formatting
import javax.imageio.ImageWriter;		// import image writer
import javax.imageio.ImageWriteParam;		// import image write parameter
import javax.imageio.stream.FileImageOutputStream; // import image output

import java.net.URL;				// import URL
import java.net.URLConnection;			// import URL connection

/**
  Get a JPEG image from an IP camera such as the Tenvis JPT3815W and lookalikes.
  The code is called as follows to read an image and store it locally as a JPEG
  file (the URL may optionally end with '/'):

  <pre>
    java GetCamera http://host:port image.jpeg
  </pre>

  @author	Kenneth J. Turner
  @version	1.0 (4th June 2012)
  <br/>		1.1 (19th October 2015)
		<ul>
		  <li>
		    instead of changing hue to intensify green, all RGB channels
		    as well as brightness and saturation can now be individually
		    changed
		  </li>
		</ul>
  <br/>		1.2 (14th April 2016)
		<ul>
		  <li>
		    brightness change now uses gamma correction to enhance dark
		    areas
		  </li>
		</ul>
  <br/>		1.2 (31st January 2019)
		<ul>
		  <li>
		    exception messages now prefixed by time and date
		  </li>
		  <li>
		    support for Tenvis TH661 added
		  </li>
		  <li>
		    constants introduced for full and thumbnail image fonts
		  </li>
		</ul>
*/
public class GetCamera {

  /** Password for camera */
  private final static String CAMERA_PASSWORD = "XXX";

  /** Tenvis JPT3815W camera */
  private final static String CAMERA_JPT3815W = "JPT3815W";

  /** Tenvis JPT3815W camera */
  private final static String CAMERA_TH661 = "TH661";

  /** Type of camera */
  private final static String CAMERA_TYPE = "XXX";

  /** Username for camera */
  private final static String CAMERA_USERNAME = "XXX";

  /** Blue colour change (additive, +/- 0..255, 0 = no change) */
  private final static int CHANGE_BLUE = +0;

  /** Brightness change (gamma factor > 0.0, 1.0 = no change) */
  private final static float CHANGE_BRIGHTNESS = 1.0f;

  /** Green colour change (additive, +/- 0..255, 0 = no change) */
  private final static int CHANGE_GREEN = +0;

  /** Red colour change (additive, +/- 0..255, 0 = no change) */
  private final static int CHANGE_RED = 0;

  /**
    Saturation change (additive if less than 0.5 else multiplicative,
    0.0 = no change)
  */
  private final static float CHANGE_SATURATION = 0.0f;

  /** Font family for full images */
  private final static String FONT_NAME_FULL = Font.MONOSPACED;

  /** Font family for full images */
  private final static String FONT_NAME_THUMB = Font.SANS_SERIF;

  /** Font size for label on full image */
  private final static int FONT_SIZE_FULL = 36;

  /** Font size for label on thumbnail image */
  private final static int FONT_SIZE_THUMB = 10;

  /** Font style for full and thumbnail images */
  private final static int FONT_STYLE = Font.BOLD;

  /** Image compression ratio (from maximum 0.0 to minimum 1.0) */
  private final static float IMAGE_COMPRESSION = 0.25f;

  /** Image date format as DAY HH:MM */
  private final static SimpleDateFormat IMAGE_DATE =
    new SimpleDateFormat("EEE HH:mm");

  /** Log date format as HH:MM:SS DD/MM/YYYY */
  private final static SimpleDateFormat LOG_DATE =
    new SimpleDateFormat("HH:mm:ss dd/MM/YYYY");

  /** Label colour */
  private final static Color LABEL_COLOUR = Color.WHITE;

  /** Label x/y offset in pixels */
  private final static int LABEL_OFFSET = 3;

  /** Thumbnail size relative to full size */
  private final static float THUMB_SCALE = 0.125f;

  /** Suffix of thumbnail image */
  private final static String THUMB_SUFFIX = "s";

  /** Width of thumbnail image */
  private final static int THUMB_WIDTH = 160;

  /** Hue-Saturation-Brightness values */
  private static float[] hsb = new float[3];

  /**
    Main program to capture an image from an IP camera into the given file.

    @param arguments	command-line arguments (camera URL, filename to create)
  */
  public static void main(String[] arguments) {
    if (arguments != null && arguments.length == 2) { // two arguments?
      try {					// try to get image
	getImage(arguments[0], arguments[1]);	// get image for named file
      }
      catch (Exception exception) {		// exception getting image?
	String timestamp = 			// generate timestamp
	  LOG_DATE.format(new Date());
	System.out.println(timestamp + " GetCamera: Exception - " +
	  exception.getMessage());
	System.exit(1);				// exit with code 1
      }
    }
    else {					// not one argument
      System.out.println("usage: GetCamera URL image_file"); // report error
      System.exit(1);				// exit with code 1
    }
  }

  /**
    Modify colour and saturation of given image, returning the new image. Code
    adapted from http://www.jhlabs.com/ip/filters.

    @param source	original image
    @return		new image
  */
  private static BufferedImage changeColourSaturation(BufferedImage source) {
    int width = source.getWidth();		// get source width
    int height = source.getHeight();		// get source height
    int imageType = source.getType();		// get source type
    WritableRaster sourceRaster =		// get source raster image
      source.getRaster();
    BufferedImage destination =			// create destination image
	new BufferedImage(source.getWidth(), source.getHeight(), imageType);
    WritableRaster destinationRaster =		// get destination raster image
      destination.getRaster();

    int[] inPixels = new int[width];		// create source row
    if (imageType == BufferedImage.TYPE_INT_ARGB) { // Alpha and RGB?
      for (int y = 0; y < height; y++) {	// go through columns
	sourceRaster.getDataElements(		// get source row
	  0, y, width, 1, inPixels);
	for (int x = 0; x < width; x++)		// go through rows
	  inPixels[x] =				// change pixel colour/satur.
	    changeColourSaturation(inPixels[x]);
	destinationRaster.setDataElements(	// set destination row
	  0, y, width, 1, inPixels);
      }
    }
    else {					// RGB
      for (int y = 0; y < height; y++) {	// go through columns
	source.getRGB(				// get source row
	  0, y, width, 1, inPixels, 0, width);
	for (int x = 0; x < width; x++)		// go through rows
	  inPixels[x] = changeColourSaturation(inPixels[x]); // change pixel sat.
	destination.setRGB(			// set destination row
	  0, y, width, 1, inPixels, 0, width);
      }
    }
    return(destination);			// return new image
  }

  /**
    Adjust colour of given RGB value by CHANGE_RED/GREEN/BLUE, its saturation by
    CHANGE_SATURATION, and its brightness by CHANGE_BRIGHTNESS. The code is
    partly based on http://www.jhlabs.com/ip/filters.

    @param rgb		current RGB value
    @return		new RGB value
  */
  private static int changeColourSaturation(int rgb) {
    int a = rgb & 0xFF000000;			// get Alpha channel
    int r = (rgb >> 16) & 0xFF;			// get red channel
    int g = (rgb >> 8) & 0xFF;			// get green channel
    int b = rgb & 0xFF;				// get blue channel
    r += CHANGE_RED;				// change red channel
    r = r < 0 ? 0 : r > 255 ? 255 : r;		// make red 0..255
    g += CHANGE_GREEN;				// change green channel
    g = g < 0 ? 0 : g > 255 ? 255 : g;		// make green 0..255
    b += CHANGE_BLUE;				// change blue channel
    b = b < 0 ? 0 : b > 255 ? 255 : b;		// make blue 0..255
    Color.RGBtoHSB(r, g, b, hsb);		// convert RGB to HSB
    float saturation = hsb[1];			// get saturation
    if (CHANGE_SATURATION < 0.5f)		// low saturation change?
      saturation += CHANGE_SATURATION;		// add saturation change
    else					// high saturation change
     saturation *= CHANGE_SATURATION;		// multiply saturation change
    hsb[1] =					// set saturation 0..1
      saturation < 0.0f ? 0.0f : saturation > 1.0f ? 1.0f : saturation;
    float brightness = hsb[2];			// get brightness
    brightness =				// apply gamma correction
      (float) Math.pow(brightness, 1.0 / CHANGE_BRIGHTNESS);
    /*
    // apply additive/multiplicative correction
    if (CHANGE_BRIGHTNESS < 0.5f)		// low brightness change?
      brightness += CHANGE_BRIGHTNESS;		// add brightness change
    else					// high saturation change
     brightness *= CHANGE_BRIGHTNESS;		// multiply brightness change
    */
   hsb[2] =					// set brightness 0..1
      brightness < 0.0f ? 0.0f : brightness > 1.0f ? 1.0f : brightness;
    rgb = Color.HSBtoRGB(hsb[0], hsb[1], hsb[2]); // HSB to RGB
    return(a | (rgb & 0xffffff));		// return Alpha and RGB
  }

  /**
    Get current image in full and thumb form, storing these in the current
    directory.

    @param urlBase	base camera URL
    @param imageName	base name for the image (without suffix)
  */
  private static void getImage(String urlBase, String imageName)
   throws IOException {
    int pos = imageName.lastIndexOf('.');	// get last '.' in name
    if (pos != -1) {				// last '.' found?
      // get raw image
      int urlBaseLlast = urlBase.length() - 1;	// get URl base last index
      if (urlBase.charAt(urlBaseLlast) == '/')	// URL base ends with '/'?
	urlBase = urlBase.substring(0, urlBaseLlast); // remove trailing '/'
      BufferedImage rawImage = getRaw(urlBase);	// get raw image
      if (rawImage == null)			// image not read?
	throw(new IOException("could not get image")); // report image not read
      int imageType = rawImage.getType();	// get image type
      if (imageType == 0)			// image type unknown?
	imageType = BufferedImage.TYPE_INT_ARGB;// assume RGB with Alpha
      int rawWidth = rawImage.getWidth();	// get raw image width
      int rawHeight = rawImage.getHeight();	// get raw image height

      // create full image
      File fullFile = new File(imageName);	// set full image file
      BufferedImage fullImage =			// get corrected full image
	changeColourSaturation(rawImage);

      // create thumbnail image
      File thumbFile =				// set thumbnail image file
	new File (imageName.substring(0, pos) + THUMB_SUFFIX +
	  imageName.substring(pos));
      int thumbWidth =				// get thumbnail width
	Math.round(THUMB_SCALE * rawWidth);
      int thumbHeight =				// get thumbnail height
	Math.round(THUMB_SCALE * rawHeight);
      BufferedImage thumbImage =		// create thumbnail image
	new BufferedImage(thumbWidth, thumbHeight, imageType);
      Graphics2D graphics =			// get graphics object
	thumbImage.createGraphics();
      graphics.drawImage(			// draw thumbnail image
	fullImage, 0, 0, thumbWidth, thumbHeight, null);
      graphics.dispose();			// dispose of graphics

      // label and save images
      writeImage(fullFile, fullImage, true);	// write full image
      writeImage(thumbFile, thumbImage, false); // write thumbnail image
    }
    else					// last '.' not found
      throw(new IOException("image name '" + imageName + "' lacks a suffix"));
  }

  /**
    Get raw camera image.

    @param urlBase	base camera URL
    @return		buffered raw image
  */
  private static BufferedImage getRaw(String urlBase) throws IOException {
    BufferedImage rawImage;
    if (CAMERA_TYPE.equals(CAMERA_JPT3815W)) {
      URL cameraURL = new URL(			// set camera URL
	urlBase + "/snapshot.cgi?user=" + CAMERA_USERNAME +
	"&pwd=" + CAMERA_PASSWORD);
      rawImage = ImageIO.read(cameraURL);	// get raw image

    }
    else if (CAMERA_TYPE.equals(CAMERA_TH661)) {
      URL cameraURL = new URL(			// set camera URL
	urlBase + "/tmpfs/snap.jpg");
      Encoder encoder = Base64.getEncoder();
      String authorisation =  new String(encoder.encode(
	(CAMERA_USERNAME + ":" + CAMERA_PASSWORD).getBytes()));
      URLConnection urlConnection = cameraURL.openConnection();
      urlConnection.setRequestProperty("Authorization",
	"Basic " + authorisation);
      rawImage =				// get raw image
	ImageIO.read(urlConnection.getInputStream());
    }
    else
      throw(new IOException("unknown camera type'" + CAMERA_TYPE + "'"));
    return(rawImage);
  }

  /**
    Label image at bottom right with day and time (e.g. "Wed 15.30").

    @param source	image source
    @param fullSize	whether full size or thumbnail
  */
  private static void labelImage(BufferedImage source, boolean fullSize) {
    String fontName = fullSize ? FONT_NAME_FULL : FONT_NAME_THUMB;
    int fontSize = fullSize ? FONT_SIZE_FULL : FONT_SIZE_THUMB;
    int fontStyle = FONT_STYLE;
    Graphics2D graphics = source.createGraphics(); // get graphics object
    String timestamp = 				// generate timestamp
      IMAGE_DATE.format(new Date());
    graphics.setColor(LABEL_COLOUR);		// set label colour
    graphics.setFont(				// set label font
      new Font(fontName, fontStyle, fontSize));
    graphics.drawString(			// draw label onto image
      timestamp, LABEL_OFFSET, source.getHeight() - LABEL_OFFSET);
    graphics.dispose();				// dispose of graphics
  }

  /**
    Label image and save it compressed.

    @param imageFile	image file
    @param source	image source
    @param fullSize	whether full size or thumbnail
    @throws		IOException
  */
  private static void writeImage(File imageFile, BufferedImage source,
   boolean fullSize) throws IOException {
    labelImage(source, fullSize);		// label image
    Iterator<ImageWriter> imageIterator =	// get JPEG image writers
      ImageIO.getImageWritersByFormatName("jpeg");
    ImageWriter imageWriter =			// get first JPEG image writer
      imageIterator.next();
    ImageWriteParam imageParameter =		// get default writer parameter
      imageWriter.getDefaultWriteParam();
    imageParameter.setCompressionMode(		// set explicit writer mode
      ImageWriteParam.MODE_EXPLICIT);
    imageParameter.setCompressionQuality(	// set writer image compression
      IMAGE_COMPRESSION);
    FileImageOutputStream imageOutput =		// get image out stream
      new FileImageOutputStream(imageFile);
    imageWriter.setOutput(imageOutput);		// set image writer output
    IIOImage image =				// set image creation
      new IIOImage(source, null, null);
    imageWriter.write(				// write image
	null, image, imageParameter);
    imageWriter.dispose();			// dispose of writer
  }

}
