/**************************************************************************

  "get_pi.c"	K. J. Turner <kjt@cs.stir.ac.uk>	02/05/01

  This program converts the Pilot database files according to the following
  command-line options:
  
    -a	"AddressDB.pdb" is converted to "<category>.addr" files
   
    -d	"DatebookDB.pdb" is converted to "Datebook"
    
    -e	"ExpenseDB.pdb" is converted to "<category>.exp" files
    
    -h	Print usage help
    
    -m	"MemoDB.pdb" is converted to "<title>.memo[s]" files
    
    -t	"ToDoDB.pdb" is converted to "ToDo"
  
  All files must be in the CURRENT directory. Any existing Address, etc.
  files will be OVERWRITTEN.

  To handle different formats from the ones generated by this program,
  customise the following functions:
  
    Address	get_addr_ent
    
    Datebook	get_appt_ent
    
    Expense	get_exp_ent
    
    Memo	get_memo_ent
    
    ToDo	get_todo_ent
  
  Inspired by "pilot-xfer" and "pilot-file" utilities by Kenneth Albanowski
  <kjahds@kjahds.com> and Pace Willisson <pace@blitz.com>.

  This is free software, distributed under the GNU Public License V2.

**************************************************************************/
          
/********************************* Globals *******************************/

#include "piconv.h"

/* make time value given structure, ensuring time offset initialised */

time_t mk_time (struct tm * timeptr)
{
  timeptr->tm_gmtoff = 0;			/* set zero offset */
  timeptr->tm_zone = "";			/* set no timezone */
  mktime (timeptr);				/* make time value */
}

/******************************* Address Book *****************************/

/*** return index for field name in address information ***/

int addr_fld_ind (struct AddressAppInfo * addr_info, char * fld_name)
{
  int fld_ind = 0;
  
  while ((fld_ind < ADDR_FLD_COUNT) &&
         strcmp (addr_info->labels[fld_ind], fld_name))
    fld_ind++;
  if (fld_ind == ADDR_FLD_COUNT) {
    fld_ind = -1;
  }
  return (fld_ind);
}

/*** append to address line, given list of entries, separator and field
     index; replace "~" by "^" by space (first/last name) or newline
     (work/home address, links) ***/

void addr_app (char * addr_line, char * entry[], int fld_ind, char * sep)
{
  char * field;					/* address field */
  char ch;					/* current character */

  addr_line += strlen (addr_line);		/* go to end of address */
  if (sep)					/* separator given? */
    strcat (addr_line, sep);			/* append it */
  if ((fld_ind != -1) &&			/* valid field index? */
      (field = entry[fld_ind])) {		/* non-empty field? */
    if (!sep && prev_fld)			/* no sep/non-empty prev? */
      strcat (addr_line, "~");			/* use "~" by default */
    addr_line += strlen (addr_line);		/* init command pointer */
    prev_fld = field;				/* update previous entry */
    while (ch = *(field++)) {			/* get character till NUL */
      if ((ch == ' ') &&			/* space to translate? */
          ((fld_ind == first_fld) ||		/* first name? */
	   (fld_ind == last_fld)))		/* last name? */
	ch = '^';				/* change to caret */
      if ((ch == '\n') &&			/* newline to translate? */
          ((fld_ind == workad_fld) ||		/* work address? */
	   (fld_ind == homead_fld) ||		/* home address? */
	   (fld_ind == link_fld)))		/* links? */
	ch = '~';				/* change to tilde */
      *(addr_line++) = ch;			/* store character */
    }
    *addr_line = '\0';				/* terminate address line */
  }
  else						/* field is empty/missing */
    prev_fld = NULL;				/* update previous entry */
}

/*** process address info to get category and field index numbers ***/

void get_addr_info (struct pi_file * pf, struct AddressAppInfo * addr_info)
{
  void * buff;					/* file buffer */
  int size;					/* data size */
  
  if (!pi_file_get_app_info (pf, &buff, &size)) { /* app info read? */
    unpack_AddressAppInfo(addr_info, buff, size); /* unpack it */
    /* get field indexes by name */
    last_fld = addr_fld_ind (addr_info, "Last name");
    first_fld = addr_fld_ind (addr_info, "First name");
    comp_fld = addr_fld_ind (addr_info, "Company");
    workph_fld = addr_fld_ind (addr_info, "Work");
    homeph_fld = addr_fld_ind (addr_info, "Home");
    fax_fld = addr_fld_ind (addr_info, "Fax");
    mobile_fld = addr_fld_ind (addr_info, "Mobile");
    email1_fld = addr_fld_ind (addr_info, "E-mail");
    workad_fld = addr_fld_ind (addr_info, "Address");
    town_fld = addr_fld_ind (addr_info, "Town");
    county_fld = addr_fld_ind (addr_info, "County");
    code_fld = addr_fld_ind (addr_info, "Post Code");
    country_fld = addr_fld_ind (addr_info, "Country");
    title_fld = addr_fld_ind (addr_info, "Title");
    email2_fld = addr_fld_ind (addr_info, "E-mail 2");
    web_fld = addr_fld_ind (addr_info, "Web");
    link_fld = addr_fld_ind (addr_info, "Links");
    homead_fld = addr_fld_ind (addr_info, "Home Addr");
    note_fld = addr_fld_ind (addr_info, "Note");
    main_fld = addr_fld_ind (addr_info, "Main");
    pager_fld = addr_fld_ind (addr_info, "Pager");
    other_fld = addr_fld_ind (addr_info, "Other");
  }
  else
    warning ("cannot read address info");
}

/*** get address entry into preferred format ***/

void get_addr_ent (struct Address * addr, int cat)
{
  char addr_line[ADDR_SIZE];			/* address entry */

  prev_fld = NULL;				/* empty previous field */
  addr_line[0] = '\0';				/* initialise address line */
  addr_app (addr_line, addr->entry, title_fld, "T");	/* title */
  addr_app (addr_line, addr->entry, first_fld, "\tp");	/* prename */
  addr_app (addr_line, addr->entry, last_fld, "\ts");	/* surname */
  addr_app (addr_line, addr->entry, workph_fld, "\tt");	/* work ph. */
  addr_app (addr_line, addr->entry, homeph_fld, "~");	/* home ph. */
  addr_app (addr_line, addr->entry, fax_fld, "~");	/* fax ph. */
  addr_app (addr_line, addr->entry, mobile_fld, "~");	/* mob. ph. */
  addr_app (addr_line, addr->entry, comp_fld, "\ta");	/* company */
  addr_app (addr_line, addr->entry, workad_fld, NULL);	/* work ad. */
  addr_app (addr_line, addr->entry, town_fld, NULL);	/* town */
  addr_app (addr_line, addr->entry, county_fld, NULL);	/* county */
  addr_app (addr_line, addr->entry, code_fld, NULL);	/* post cd. */
  addr_app (addr_line, addr->entry, country_fld, NULL);	/* country */
  addr_app (addr_line, addr->entry, homead_fld, "\tA");	/* home ad. */
  addr_app (addr_line, addr->entry, email1_fld, "\te");	/* email1 */
  addr_app (addr_line, addr->entry, email2_fld, "~");	/* email2 */
  addr_app (addr_line, addr->entry, web_fld, "~");	/* web ad. */
  addr_app (addr_line, addr->entry, link_fld, "\tl");	/* links */
  /* miscellaneous entries are appended to links */
  addr_app (addr_line, addr->entry, note_fld, NULL);	/* note */
  addr_app (addr_line, addr->entry, main_fld, NULL);	/* main ph. */
  addr_app (addr_line, addr->entry, pager_fld, NULL);	/* page ph. */
  addr_app (addr_line, addr->entry, other_fld, NULL);	/* telex */
  strcat (addr_line, "\n");			/* append newline */
  fputs (addr_line, cat_file[cat]);		/* write address line */
}

/*** get address entries into preferred format ***/

void get_addr_ents (struct pi_file * pf, struct DBInfo * info,
  struct AddressAppInfo * addr_info)
{
  FILE * af;					/* address file */
  struct Address addr;				/* address */
  int size;					/* data size */
  int i;					/* counter */
  int no_ents;					/* number of entries */
  int no_ent;					/* number of entry */
  int attrs, cat, cur_cat;			/* attributes, category */
  int fld_ind;					/* field index */
  int non_space;				/* 1 = string non-space */
  unsigned long uid;				/* user id */
  char appt_name[LINE_SIZE];			/* address file name */
  char cat_name[CAT_SIZE];			/* category name */
  char file_mode[2];				/* file mode */
  char * cp;					/* character pointer */
  void * buff;					/* file buffer */
  char ch;					/* current character */
  char * cp1, * cp2;				/* character pointers */

  /* open output address files */
  for (fld_ind = 0; fld_ind < CAT_COUNT; fld_ind++) { /* for all cats */
    if (*(addr_info->category.name[fld_ind]) != '\0') { /* cat. name given? */
      cp1 = addr_info->category.name[fld_ind];	/* set cat. name pointer */
      cp2 = appt_name;				/* set file name pointer */
      i = 0;					/* initialise name count */
      non_space = 0;				/* assume string spaces */
      while (((ch = *(cp1++)) != '\0') &&	/* copy till NUL/full */
	      (i++ < CAT_SIZE)) {
	if ((ch == ' ') || (ch == '/'))		/* space/slash to "_" */
	  ch = '_';
	if (!isspace (ch))			/* character is non-space? */
	  non_space = 1;			/* mark string as non-space */
	*(cp2++) = ch;				/* store character */
      }
      *cp2 = '\0';				/* terminate file name */
      if (!non_space) {				/* name white space? */
	cp2 = appt_name;			/* re-initialise pointer */
	strcpy (cp2, "Address");		/* default name "Address" */
      }
      strcat (cp2, ".addr");			/* append address suffix */
      if ((af = fopen (appt_name, "w")) == NULL ) { /* open file failed? */
	sprintf (message, "cannot open %s", appt_name);
	fatal (message);
      }
    }
    else					/* no category name */
      af = NULL;				/* set no file handle */
    cat_file[fld_ind] = af;			/* set address file handle */
  }

  pi_file_get_entries (pf, &no_ents);		/* get number of entries */
  if (!(info->flags & dlpDBFlagResource)) {	/* process entries */
    for (no_ent = 0; no_ent < no_ents; no_ent++) {
      if (pi_file_read_record (pf, no_ent, &buff, &size,
           &attrs, &cat, &uid) < 0) {	/* get address entry */
	sprintf (message, "cannot read entry %d", no_ent);
	fatal (message);
      }
      if (!(attrs & dlpRecAttrDeleted) &&	/* not deleted/archived */
          !(attrs & dlpRecAttrArchived)) {
	unpack_Address (&addr, buff, size);	/* extract entry fields */
	get_addr_ent (&addr, cat);		/* get address entry */
      }
    }
  }

  /* close output address files */
  for (fld_ind = 0; fld_ind < CAT_COUNT; fld_ind++) { /* for all cats */
    if (cat_file[fld_ind])			/* file handle was set? */
      fclose (cat_file[fld_ind]);		/* close it */
  }
}

/*** read and format addresses ***/

void from_address ()
{
  struct pi_file * pf;				/* pilot file */
  char * pi_name;				/* pilot file name */
  struct DBInfo info;				/* database info */
  struct AddressAppInfo addr_info;		/* address info */

  printf ("%s: converting from Pilot address book\n", progname);
  pi_name = "AddressDB.pdb";			/* set pilot file name */
  if (pf = pi_file_open (pi_name)) {		/* open pilot file? */
    if (pi_file_get_info (pf, &info) >= 0) {	/* get file info */
      get_addr_info (pf, &addr_info);		/* extract addr info */
      get_addr_ents (pf, &info, &addr_info);	/* extract addr entries */
    }
    else					/* cannot get file info */
      warning ("cannot get info");
    pi_file_close (pf);				/* close pilot file */
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", pi_name);
    warning (message);
  }
}

/******************************* Date Book *****************************/

/*** process datebook info (no fields actually needed) ***/

void get_appt_info (struct pi_file * pf,
  struct AppointmentAppInfo * appt_info)
{
  void * buff;					/* file buffer */
  int size;					/* data size */
  
  if (!pi_file_get_app_info (pf, &buff, &size)) { /* app info read? */
    unpack_AppointmentAppInfo (appt_info, buff, size); /* unpack it */
  }
  else
    warning ("cannot read datebook info");
}

/*** get datebook entry into preferred format ***/

void get_appt_ent (struct Appointment * appt, char * appt_line, int secret)
{
  struct tm * alarm_tm;				/* alarm struct */
  time_t alarm_time, adv;			/* alarm times */
  char appt_fld[LINE_SIZE];			/* appointment fields */

  sprintf (appt_fld, "%s %02d %04d\t",		/* set month day year */
    mon_name[appt->begin.tm_mon], appt->begin.tm_mday,
      appt->begin.tm_year + 1900);
  strcpy (appt_line, appt_fld);			/* set initial fields */
  if (!appt->event) {				/* timed event? */
    sprintf (appt_fld, "%02d.%02d", 		/* append start time */
      appt->begin.tm_hour, appt->begin.tm_min);
    strcat (appt_line, appt_fld);
    if ((appt->begin.tm_hour != 		/* end hour differs? */
	  appt->end.tm_hour) ||
	(appt->begin.tm_min !=			/* end minute differs? */
	  appt->end.tm_min)) {
      sprintf (appt_fld,			/* append end time */
	"-%02d.%02d\t", appt->end.tm_hour, appt->end.tm_min);
      strcat (appt_line, appt_fld);
    }
    else					/* start time = end time */
      strcat (appt_line, "\t\t");		/* leave gap */
  }
  else						/* untimed event */
    strcat (appt_line, "\t\t");			/* leave gap */
  if (secret)					/* record is secret? */
    strcat (appt_line, "@");			/* append secret marker */
  strcat (appt_line, appt->description);	/* append description */
  if (appt->note) {				/* note present? */
    strcat (appt_line, " (");			/* append note in (...) */
    strcat (appt_line, appt->note);
    strcat (appt_line, ")");
  }
  if (appt->alarm) {				/* alarm time set? */
    alarm_time = mk_time (&appt->begin);	/* set start time */
    if (appt->advanceUnits == advMinutes)	/* advance in minutes? */
      adv = appt->advance * 60;			/* turn mins to secs */
else if (appt->advanceUnits == advHours) 	/* advance in hours? */
      adv = appt->advance * HOUR_SECS;		/* turn hours to secs */
else if (appt->advanceUnits == advDays) 	/* advance in days? */
      adv = appt->advance * DAY_SECS;		/* turn days to secs */
else {						/* unknown advance type */
      sprintf (message, "unknown alarm advance %d", appt->advance);
      warning (message);
    }
    alarm_time -= adv;				/* reduce by advance */
    alarm_tm = localtime (&alarm_time);		/* convert to structure */
    sprintf (appt_fld, "\t%02d.%02d",		/* append alarm time */
      alarm_tm->tm_hour, alarm_tm->tm_min);
    strcat (appt_line, appt_fld);
  }
  strcat (appt_line, "\n");			/* finish appointment line */
}

/*** get datebook entries into preferred format ***/

void get_appt_ents (struct pi_file * pf, struct DBInfo * info,
  struct AppointmentAppInfo * appt_info)
{
  FILE *df;					/* appointment output file */
  struct Appointment appt;			/* appointment */
  struct tm * rep_tm;				/* repeat struct */
  time_t rep_time, end_time;			/* repeat beg/end times */
  char appt_line[APPT_SIZE];			/* appointment entry */
  char * cp1;					/* character pointer */
  void * buff;					/* file buffer */
  int size;					/* data size */
  int no_ents;					/* number of entries */
  int no_ent;					/* number of entry */
  int attrs, cat;				/* attributes, category */
  unsigned long uid;				/* user id */
  int appt_rep;					/* repeat indicator */	

  cp1 = "Datebook";				/* set output file name */
  if (df = fopen (cp1, "w")) {			/* open datebook file? */
    pi_file_get_entries (pf, &no_ents);		/* get number of entries */
    if (!(info->flags & dlpDBFlagResource)) {	/* process entries */
      for (no_ent = 0; no_ent < no_ents; no_ent++) {
	if (pi_file_read_record (pf, no_ent, &buff, &size, &attrs, &cat, &uid)
	      < 0) {				/* get appointment entry */
	  sprintf (message, "cannot read entry %d", no_ent);
	  fatal (message);
	}
	if (!(attrs & dlpRecAttrDeleted) &&	/* not deleted/archived */
	    !(attrs & dlpRecAttrArchived)) {
	  unpack_Appointment (&appt, buff, size); /* extract entry fields */
	  if (appt.repeatForever) {		/* forever? take as 1 year */
	    appt.repeatEnd.tm_year = appt.begin.tm_year + 1;
	    appt.repeatEnd.tm_mon = appt.begin.tm_mon;
	    appt.repeatEnd.tm_mday = appt.begin.tm_mday;
	    appt.repeatEnd.tm_isdst = appt.begin.tm_isdst;
	    appt.repeatEnd.tm_gmtoff = 0; appt.repeatEnd.tm_zone = "";
	  }
	  appt.repeatEnd.tm_hour = appt.begin.tm_hour;
	  appt.repeatEnd.tm_min = appt.begin.tm_min;
	  appt.repeatEnd.tm_sec = appt.begin.tm_sec;
	  end_time = mk_time (&appt.repeatEnd);	/* set repeat end time */
	  do {					/* output entry */
	    get_appt_ent (&appt, appt_line,	/* get/output appt entry */
	      attrs & dlpRecAttrSecret);
	    fputs (appt_line, df);		/* output appointment */
	    appt_rep = appt.repeatType;		/* extract repeat indicator */
	    if (appt_rep) {			/* appointment repeats? */
	      rep_time = mk_time (&appt.begin);	/* set current time */
	      switch (appt_rep) {
		case repeatDaily:		/* add one day */
		  rep_time += appt.repeatFrequency * DAY_SECS;
		  break;
		case repeatWeekly:		/* add one week */
		  rep_time += appt.repeatFrequency * WEEK_SECS;
		  break;
		case repeatMonthlyByDay:	/* same day each month */
		  sprintf (message, "no support for repeat by day");
		  warning (message);
		  appt_rep = 0;			/* reset repeat type */
		  continue;			/* leave loop */
		case repeatMonthlyByDate:	/* add one month */
		  appt.begin.tm_mon++;
	          rep_time = mk_time (&appt.begin);
		  break;
		case repeatYearly:		/* add one Year */
		  appt.begin.tm_year++;
	          rep_time = mk_time (&appt.begin);
		  break;
	      }
	      if (rep_time <= end_time) {	/* not past repeat end? */
		rep_tm = localtime (&rep_time);	/* update current begin */
		appt.begin.tm_mday = rep_tm->tm_mday;
		appt.begin.tm_mon = rep_tm->tm_mon;
		appt.begin.tm_year = rep_tm->tm_year;
	      }
	      else				/* past repeat end */
		appt_rep = 0;			/* force end of repeat */
	    }
	  } while (appt_rep);			/* output till all repeats */
	}
      }
    }
    fclose (df);				/* close output file */
  }
  else {					/* cannot open output file */
      sprintf (message, "cannot write %s", cp1);
      warning (message);
  }
}

void from_datebook ()
{
  struct pi_file * pf;				/* pilot file */
  char * pi_name;				/* pilot file name */
  struct DBInfo info;				/* database info */
  struct AppointmentAppInfo appt_info;		/* appointment info */

  printf ("%s: converting from Pilot date book\n", progname);
  pi_name = "DatebookDB.pdb";			/* set pilot file name */
  if (pf = pi_file_open (pi_name)) {		/* open pilot file? */
    if (pi_file_get_info (pf, &info) >= 0) {	/* get file info */
      get_appt_info (pf, &appt_info);		/* extract appt info */
      get_appt_ents (pf, &info, &appt_info);	/* extract appt entries */
    }
    else					/* cannot get file info */
      warning ("cannot get info");
    pi_file_close (pf);				/* close pilot file */
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", pi_name);
    warning (message);
  }
}

/******************************* Expenses *****************************/

/*** process expense info ***/

void get_exp_info (struct pi_file * pf,
  struct ExpenseAppInfo * exp_info)
{
  FILE * ef;					/* output expense file */
  void * buff;					/* file buffer */
  int size;					/* data size */
  
  if (!pi_file_get_app_info (pf, &buff, &size)) { /* app info read? */
    unpack_ExpenseAppInfo (exp_info, buff, size); /* unpack it */
  }
  else
    warning ("cannot read expense info");
}

/*** get expense entry into preferred format ***/

void get_exp_ent (struct ExpenseAppInfo * exp_info, struct Expense * exp,
  int cat)
{
  int first_qual;				/* 1 = 1st exp. qualifier */
  char exp_line[EXP_SIZE];			/* expense entry */
  char ch;					/* current character */
  char * cp1, * cp2;				/* character pointers */

  sprintf (exp_line, "%02d-%s-%04d\t",		/* expense date */
    exp->date.tm_mday, mon_name[exp->date.tm_mon], exp->date.tm_year + 1900);
  strcat (exp_line, exp_type[exp->type]);	/* append expense type */
  if (exp_pay[exp->payment] || exp->vendor	/* expense qualifier? */
	|| exp->city || exp->attendees) {
    first_qual = 1;				/* note as first qualifier */
    strcat (exp_line, " (");			/* begin qualifier */
    if (exp_pay[exp->payment]) {		/* payment type? */
      if (first_qual)				/* no previous qualifier? */
	first_qual = 0;				/* note not now first */
      else					/* previous qualifier */
	strcat (exp_line, ", ");		/* add sub-field separator */
      strcat (exp_line, exp_pay[exp->payment]); /* append payment type */
    }
    if (exp->vendor) {				/* vendor? */
      if (first_qual)				/* no previous qualifier? */
	first_qual = 0;				/* note not now first */
      else					/* previous qualifier */
	strcat (exp_line, ", ");		/* add sub-field separator */
      strcat (exp_line, exp->vendor);		/* append vendor */
    }
    if (exp->city) {				/* city? */
      if (first_qual)				/* no previous qualifier? */
	first_qual = 0;				/* note not now first */
      else					/* previous qualifier */
	strcat (exp_line, ", ");		/* add sub-field separator */
      strcat (exp_line, exp->city);		/* append city */
    }
    if (exp->attendees) {			/* attendees? */
      if (first_qual)				/* no previous qualifier? */
	first_qual = 0;				/* note not now first */
      else					/* previous qualifier */
	strcat (exp_line, ", ");		/* add sub-field separator */
      cp1 = exp->attendees;			/* set attendees pointer */
      cp2 = exp_line + strlen (exp_line);	/* set exp. line pointer */
      while ((ch = *(cp1++)) != '\0') {		/* copy till NUL */
	if ((ch == '\n'))			/* convert newline to space */
	  ch = ' ';
	*(cp2++) = ch;				/* store character */
      }
      *cp2 = '\0';				/* terminate attendees */
    }
    strcat (exp_line, ")");			/* end qualification */
  }
  strcat (exp_line, "\t");			/* append field separator */
  if (exp->currency < EXP_CURR_MAX)		/* standard currency? */
    strcat (exp_line, exp_curr[exp->currency]);	/* append currency */
  else						/* custom currency */
    strcat (exp_line,				/* append custom symbol */
      exp_info->currencies[exp->currency - EXP_CURR_MAX].symbol);
  strcat (exp_line, "\t");			/* append field separator */
  if (exp->amount)				/* non-null amount? */
    strcat (exp_line, exp->amount);		/* append exp. amount */
  strcat (exp_line, "\t");			/* append field separator */
  if (exp->note) {				/* note? */
    strcat (exp_line, "[");			/* begin note */
    cp1 = exp->note;				/* set note pointer */
    cp2 = exp_line + strlen (exp_line);		/* set exp. line pointer */
    while ((ch = *(cp1++)) != '\0') {		/* copy till NUL */
      if ((ch == '\n'))				/* convert newline to space */
	ch = ' ';
      *(cp2++) = ch;				/* store character */
    }
    *cp2 = '\0';				/* terminate note */
    strcat (exp_line, "]");			/* end note */
  }
  strcat (exp_line, "\n");			/* terminate expense line */
  fputs (exp_line, cat_file[cat]);		/* write expense line */
}

/*** get expense entries into preferred format ***/

void get_exp_ents (struct pi_file * pf, struct DBInfo * info,
  struct ExpenseAppInfo * exp_info)
{
  FILE *ef;					/* expense output file */
  struct Expense exp;				/* expense */
  struct tm * exp_tm;				/* expense date */
  void * buff;					/* file buffer */
  int size;					/* data size */
  int no_ents;					/* number of entries */
  int no_ent;					/* number of entry */
  int attrs, cat;				/* attributes, category */
  unsigned long uid;				/* user id */
  int i;					/* counter */
  int fld_ind;					/* field index */
  int non_space;				/* 1 = string non-space */
  char exp_name[LINE_SIZE];			/* expense file name */
  char ch;					/* current character */
  char * cp1, * cp2;				/* character pointers */

  /* open output expense files */
  for (fld_ind = 0; fld_ind < CAT_COUNT; fld_ind++) { /* for all cats */
    if (*(exp_info->category.name[fld_ind]) != '\0') { /* cat. name given? */
      cp1 = exp_info->category.name[fld_ind];	/* set cat. name pointer */
      cp2 = exp_name;				/* set file name pointer */
      i = 0;					/* initialise name count */
      non_space = 0;				/* assume string spaces */
      while (((ch = *(cp1++)) != '\0') &&	/* copy till NUL/full */
	      (i++ < CAT_SIZE)) {
	if ((ch == ' ') || (ch == '/'))		/* space/slash to "_" */
	  ch = '_';
	if (!isspace (ch))			/* character is non-space? */
	  non_space = 1;			/* mark string as non-space */
	*(cp2++) = ch;				/* store character */
      }
      *cp2 = '\0';				/* terminate file name */
      if (!non_space) {				/* name white space? */
	cp2 = exp_name;				/* re-initialise pointer */
	strcpy (cp2, "Expense");		/* default name "Expense" */
      }
      strcat (cp2, ".exp");			/* append expense suffix */
      if ((ef = fopen (exp_name, "w")) == NULL ) { /* open file failed? */
	sprintf (message, "cannot open %s", exp_name);
	fatal (message);
      }
    }
    else					/* no category name */
      ef = NULL;				/* set no file handle */
    cat_file[fld_ind] = ef;			/* set expense file handle */
  }

  /* process pilot expense entries */
  pi_file_get_entries (pf, &no_ents);		/* get number of entries */
  if (!(info->flags & dlpDBFlagResource)) {	/* process entries */
    for (no_ent = 0; no_ent < no_ents; no_ent++) {
      if (pi_file_read_record (pf, no_ent, &buff, &size, &attrs, &cat, &uid)
	    < 0) {				/* get expense entry */
	sprintf (message, "cannot read entry %d", no_ent);
	fatal (message);
      }
      if (!(attrs & dlpRecAttrDeleted) &&	/* not deleted/archived */
	  !(attrs & dlpRecAttrArchived)) {
	unpack_Expense (&exp, buff, size);	/* extract entry fields */
	get_exp_ent (exp_info, &exp, cat);	/* get expense entry */
      }
    }
  }
  
  /* close output expense files */
  for (fld_ind = 0; fld_ind < CAT_COUNT; fld_ind++) { /* for all cats */
    if (cat_file[fld_ind])			/* file handle was set? */
      fclose (cat_file[fld_ind]);		/* close it */
  }
}

/*** read and format expenses ***/

void from_expense ()
{
  struct DBInfo info;				/* database info */
  struct ExpenseAppInfo exp_info;		/* expense info */
  struct pi_file * pf;				/* pilot file */
  char * pi_name;				/* pilot file name */

  printf ("%s: converting from Pilot expenses\n", progname);
  pi_name = "ExpenseDB.pdb";			/* set pilot file name */
  if (pf = pi_file_open (pi_name)) {		/* open pilot file? */
    if (pi_file_get_info (pf, &info) >= 0) {	/* get file info */
      get_exp_info (pf, &exp_info);		/* extract exp. info */
      get_exp_ents (pf, &info, &exp_info);	/* extract exp. entries */
    }
    else					/* cannot get file info */
      warning ("cannot get info");
    pi_file_close (pf);				/* close pilot file */
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", pi_name);
    warning (message);
  }
}

/******************************* Memo Pad *****************************/

/*** process memo info to get field index numbers ***/

void get_memo_info (struct pi_file * pf, struct MemoAppInfo * memo_info)
{
  void *buff;					/* file buffer */
  int size;					/* data size */
  
  if (!pi_file_get_app_info (pf, &buff, &size)) { /* app info read? */
    unpack_MemoAppInfo(memo_info, buff, size);	/* unpack it */
  }
  else
    warning ("cannot read memo info");
}

/*** get memo entry into preferred format ***/

void get_memo_ent (struct Memo * memo, char * cat_name, int attrs, int no_ent)
{
  FILE * mf;					/* memo file */
  int non_space;				/* 1 = string non-space */
  int i;					/* counter */
  char memo_name[LINE_SIZE];			/* memo line */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */

  if (!strcmp (cat_name, "Business")) {		/* business category? */
    memo_name[0] = '\0';			/* empty memo name */
  }
  else {					/* not business */
    strcpy (memo_name, cat_name);		/* name starts with cat. */
    strcat (memo_name, "-");			/* append dash */
  }
  i = strlen (memo_name);			/* initialise name length */
  cp1 = memo->text; cp2 = memo_name + i;	/* initialise pointers */
  non_space = 0;				/* assume string spaces */
  while (((ch = *(cp1++)) != '\n') &&		/* copy till NL/NUL/full */
	  (ch != '\0') && (i++ < MEMO_NAME_SIZE)) {
    if ((ch == ' ') || (ch == '/'))		/* space/slash to "_" */
      ch = '_';
    if (!isspace (ch))				/* character is non-space? */
      non_space = 1;				/* mark string as non-space */
    *(cp2++) = ch;				/* store character */
  }
  *cp2 = '\0';					/* terminate memo line */
  if (!non_space) {				/* first line white space? */
    cp2 = memo_name;				/* re-initialise pointer */
    sprintf (cp2, "memo%d", no_ent);		/* default name "memo<n>" */
  }
  strcat (cp2, ".memo");			/* add suffix */
  cp2 += strlen (cp2);				/* to end of memo name */
  if (attrs & dlpRecAttrSecret) {		/* private memo? */
    *(cp2++) = 's';				/* note as secret */
    *cp2 = '\0';				/* terminate suffix */
  }
  if (mf = fopen (memo_name, "w")) {		/* open memo file? */
    fputs (memo->text, mf);			/* output memo text */
    fclose (mf);				/* close memo file */
  }
  else {
    sprintf (message, "cannot write %s", memo_name);
    fatal (message);
  }
}

/*** get memo entries into preferred format ***/

void get_memo_ents (struct pi_file * pf, struct DBInfo * info,
  struct MemoAppInfo * memo_info)
{
  struct Memo memo;				/* memo */
  char cat_name[CAT_SIZE];			/* category name */
  void * buff;					/* file buffer */
  int size;					/* data size */
  int no_ents;					/* number of entries */
  int no_ent;					/* number of entry */
  int attrs, cat;				/* attributes, category */
  unsigned long uid;				/* user id */

  pi_file_get_entries (pf, &no_ents);		/* get number of entries */
  if (!(info->flags & dlpDBFlagResource)) {	/* process entries */
    for (no_ent = 0; no_ent < no_ents; no_ent++) {
      if (pi_file_read_record (pf, no_ent, &buff, &size, &attrs, &cat, &uid)
	    < 0) {				/* get memo entry */
	sprintf (message, "cannot read entry %d", no_ent);
	fatal (message);
      }
      if (!(attrs & dlpRecAttrDeleted) &&	/* not deleted/archived */
          !(attrs & dlpRecAttrArchived)) {
        strcpy (cat_name, memo_info->category.name[cat]); /* get cat. name */
	unpack_Memo (&memo, buff, size);	/* extract entry fields */
	get_memo_ent (&memo, cat_name, attrs, no_ent); /* get memo entry */
      }
    }
  }
}

/*** read and format memos ***/

void from_memo ()
{
  struct pi_file * pf;				/* pilot file */
  char * pi_name;				/* pilot file name */
  struct DBInfo info;				/* database info */
  struct MemoAppInfo memo_info;			/* memo info */

  printf ("%s: converting from Pilot memo book\n", progname);
  pi_name = "MemoDB.pdb";			/* set pilot file name */
  if (pf = pi_file_open (pi_name)) {		/* open pilot file? */
    if (pi_file_get_info (pf, &info) >= 0) {	/* get file info */
      get_memo_info (pf, &memo_info);		/* extract memo info */
      get_memo_ents (pf, &info, &memo_info);	/* extract memo entries */
    }
    else					/* cannot get file info */
      warning ("cannot get info");
    pi_file_close (pf);				/* close pilot file */
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", pi_name);
    warning (message);
  }
}

/******************************* To Dos *****************************/

/*** process todo info to get field index numbers ***/

void get_todo_info (struct pi_file * pf, struct ToDoAppInfo * todo_info)
{
  void *buff;					/* file buffer */
  int size;					/* data size */
  
  if (!pi_file_get_app_info (pf, &buff, &size)) { /* app info read? */
    unpack_ToDoAppInfo(todo_info, buff, size);	/* unpack it */
  }
  else
    warning ("cannot read todo info");
}

/*** get todo entry into preferred format, replacing newline by "~" ***/

void get_todo_ent (struct ToDo * todo, char * cat_name, int attrs, FILE * tf)
{
  int ii;					/* counter */
  char todo_line[TODO_SIZE];			/* todo line */
  char * cp1, * cp2;				/* character pointers */
  char ch;					/* current character */

  cp2 = todo_line;				/* initialise line pointer */
  if (todo->complete) {				/* item is done? */
    *(cp2++) = '*';				/* add "*" (done) */
    *cp2 = '\0';				/* terminate category */
  }
  else						/* item not done */
    *(cp2) = '\0';				/* empty line */
  ch = *cat_name; 				/* get first cat. letter */
  if (ch != 'B') {				/* not business? */
    *(cp2++) = ch;				/* add first cat. letter */
    *cp2 = '\0';				/* terminate category */
  }
  if (attrs & dlpRecAttrSecret) {		/* private memo? */
    *(cp2++) = 'S';				/* note as secret */
    *cp2 = '\0';				/* terminate suffix */
  }
  sprintf (cp2, "\t%d\t", todo->priority);	/* output priority */
  cp2 += strlen (cp2);				/* to end of line so far */
  if (todo->indefinite)				/* no due date? */
    strcat (cp2, "\t\t");			/* append tab separators */
  else {					/* append due date */
    sprintf (cp2, "%s %02d %04d\t",		/* set due month day year */
      mon_name[todo->due.tm_mon], todo->due.tm_mday, todo->due.tm_year + 1900);
  }
  cp2 += strlen (cp2);				/* to end of line so far */
  cp1 = todo->description;			/* initialise descr pointer */
  while (ch = *(cp1++)) {			/* get character till NUL */
    if (ch == '\n')				/* newline to translate? */
      ch = '~';					/* change to tilde */
    *(cp2++) = ch;				/* store character */
  }
  *cp2 = '\0';					/* terminate description */
  cp1 = todo->note;				/* initialise note pointer */
  if (*cp1 != '\0') {				/* note not empty? */
    strcat (cp2, "~(");				/* open newline/parenthesis */
    cp2 += strlen (cp2);			/* to end of line so far */
    while (ch = *(cp1++)) {			/* get character till NUL */
      if (ch == '\n')				/* newline to translate? */
	ch = '~';				/* change to tilde */
      *(cp2++) = ch;				/* store character */
    }
    *(cp2++) = ')';				/* close parenthesis */
    *cp2 = '\0';				/* terminate note */
  }
  strcat (cp2, "\n");				/* append newline */
  fputs (todo_line, tf);			/* output line to file */
}

/*** get todo entries into preferred format ***/

void get_todo_ents (struct pi_file * pf, struct DBInfo * info,
  struct ToDoAppInfo * todo_info)
{
  FILE * tf;					/* todo file */
  struct ToDo todo;				/* todo */
  char cat_name[CAT_SIZE];			/* category name */
  char * cp;					/* character pointers */
  void * buff;					/* file buffer */
  int size;					/* data size */
  int no_ents;					/* number of entries */
  int no_ent;					/* number of entry */
  int attrs, cat;				/* attributes, category */
  unsigned long uid;				/* user id */

  cp = "ToDo";					/* set output todo filename */
  if (tf = fopen (cp, "w")) {			/* open todo file? */
    fputs ("Status\tPri\tDue\t\tDescription\n\n", tf);
    pi_file_get_entries (pf, &no_ents);		/* get number of entries */
    if (!(info->flags & dlpDBFlagResource)) {	/* process entries */
      for (no_ent = 0; no_ent < no_ents; no_ent++) {
	if (pi_file_read_record (pf, no_ent, &buff, &size, &attrs, &cat, &uid)
	      < 0) {				/* get todo entry */
	  sprintf (message, "cannot read entry %d", no_ent);
	  fatal (message);
	}
	if (!(attrs & dlpRecAttrDeleted) &&	/* not deleted/archived */
	    !(attrs & dlpRecAttrArchived)) {
	  strcpy (cat_name, todo_info->category.name[cat]); /* get cat. name */
	  unpack_ToDo (&todo, buff, size);	/* extract entry fields */
	  get_todo_ent (&todo, cat_name, attrs, tf); /* get todo entry */
	}
      }
    }
    fclose (tf);				/* close todo file */
  }
  else {					/* cannot open todo file */
    sprintf (message, "cannot open %s", cp);
    fatal (message);
  }
}

/*** read and format todos ***/

void from_todo ()
{
  struct pi_file * pf;				/* pilot file */
  char * pi_name;				/* pilot file name */
  struct DBInfo info;				/* database info */
  struct ToDoAppInfo todo_info;			/* todo info */

  printf ("%s: converting from Pilot todo book\n", progname);
  pi_name = "ToDoDB.pdb";			/* set pilot file name */
  if (pf = pi_file_open (pi_name)) {		/* open pilot file? */
    if (pi_file_get_info (pf, &info) >= 0) {	/* get file info */
      get_todo_info (pf, &todo_info);		/* extract todo info */
      get_todo_ents (pf, &info, &todo_info);	/* extract todo entries */
    }
    else					/* cannot get file info */
      warning ("cannot get info");
    pi_file_close (pf);				/* close pilot file */
  }
  else {					/* cannot open pilot file */
    sprintf (message, "cannot open %s", pi_name);
    warning (message);
  }
}

/******************************* Main Program *****************************/

int main (int argc, char **argv)
{
  struct pi_file *pf;				/* pilot file handle */
  char name[LINE_SIZE];				/* pilot file name */
  struct DBInfo info;				/* database information */
  int arg_ind;					/* argument number */
  int opt;					/* option */
  
  progname = argv[0];				/* get program name/options */
  while ((opt = getopt (argc, argv, "adehmt")) != EOF) {
    switch (opt) {
      case 'a':					/* address book */
	from_address ();			/* convert address book */
	break;
      case 'd':					/* date book */
	from_datebook ();			/* convert date book */
	break;
      case 'h':					/* help */
	usage ();
	break;
      case 'e':					/* expenses */
	from_expense ();			/* convert expenses */
	break;
      case 'm':					/* memo pad */
	from_memo ();				/* convert memo pad */
	break;
      case 't':					/* todo list */
	from_todo ();				/* convert todo list */
	break;
      default:					/* unrecognised option */
	usage ();
    }
  }
  if (optind != argc) {				/* excess arguments? */
    warning ("only options may be given");
    usage ();
  }
  return (0);
}

/*************************************************************************/
