%{

/*****************************************************************

sdltex.l	K. J. Turner	111/05/02

This "lex" program reads SDL/PR text (which is assumed to be syntactically
valid) from the standard input, and writes it to the standard output in a
form suitable for processing by LaTeX. The flag "-n" causes output lines to
be numbered at the right.

The program translates initial tab and space characters on a line into LaTeX
horizontal space (\hspace). This assumes that indentation will be by small
(e.g. 2) units of space. Because LaTeX limits the number of tab stops in a
"tabbing" environment to around a dozen, it is not possible to have tab
stops every space or two.

Tabs and multiple spaces embedded in a line are translated into LaTeX tab
commands (\>). This assumes that embedded tabs and multiple spaces are
likely to be used to format comments and the like, for which a coarser
degree of control over spacing is acceptable.

Since the output is assumed to be embedded in a LaTeX "tabbing" environment,
a LaTeX newline command (\\) is appended to each line.

SDL 92 keywords are converted to bold (\bf). Keywords are recognised only in
lower case or with capitalisation of each word (e.g. "EndProcess", "FPar").
LaTeX special characters are escaped. (The keywords "end" and "route" are
accepted on their own.)

The program recognises "/" followed by "*" as beginning comments, "*"
followed by "/" as ending comments, and "'" as beginning or ending a
string.  Comments may not be nested. Comment delimiters are treated
literally inside strings, and string delimiters are treated as comments
inside comments. SDL keywords are not emboldened inside comments or strings.
To preserve the appearance of strings, embedded tabs and multiple spaces are
converted into tabs. This is only a rough approximation because of the
proportional spacing.

*****************************************************************/

#define Tab 8		/* number of characters between tab positions */

#define Spc 0.5		/* number "em"s for horizontal space unit */

#define Prog "sdltex"	/* program name */

int IpPos = 0;		/* current input position (0 = first) */

int OpPos = 0;		/* current output position (0 = first) */

int TabPos = 0;		/* current tab position (0 = first) */

int InComment = 0;	/* indicates whether inside a comment (1 =
			   yes) */
int InString = 0;       /* indicates whether inside a string (1 =
			   yes) */
int LineNum = 0;	/* current line number - line numbering off by
			   default */

/* output current text */

void echo ();

/*  At the start of a line make good the difference between the input
    and output positions by using "\hspace". Otherwise, round the input
    position up to next tab stop, and output LaTeX tabs to bring the
    output position up to the input position. */

void Input () {
  int NumTabs;

  if (OpPos < IpPos) {				/* positions disagree? */
    if (OpPos == 0) {				/* start of line? */
      OpPos = IpPos;
      printf ("\\hspace{%.1fem}", Spc * IpPos);
    }
    else {
	NumTabs = (IpPos - 1) / Tab + 1;	/* next tab stop */
	OpPos = Tab * NumTabs;
	while (TabPos < NumTabs) {
	  printf ("\\>"); TabPos++;
	}
    }
  }
}

/* Output current identifier string, updating input and output
   positions. Escape special LaTeX characters as required. */

void OutputId () {
  int Ind;
  char Ch;

  for (Ind = 0; Ind < yyleng; Ind++) {
    Ch = yytext [Ind];
    if (Ch == '#' || Ch == '$' || Ch == '&' || Ch == '%' || Ch == '{' || Ch == '}') {
      putchar ('\\');
      putchar (Ch);
    }
    else if (Ch == '<' || Ch == '>' || Ch == '|' || ((Ch == '-') && !InComment)) {
      putchar ('$');
      putchar (Ch);
      putchar ('$');
    }
    else if (Ch == '~' || Ch == '^') {
      putchar ('\\');
      putchar (Ch);
      putchar ('{');
      putchar ('}');
    }
    else if (Ch == '_') {
      putchar ('\\');
      putchar (Ch);
      putchar ('\\');
      putchar (',');
    }
    else if (Ch == '\\') {
      printf ("$\\backslash$");
    }
    else putchar (Ch);
  }
  IpPos += yyleng;
  OpPos += yyleng;
}

/* Output current symbol string, updating input and output positions.
   Escape special LaTeX characters as required, and use special symbols
   where necessary. */

void OutputSym () {
  if ((!strcmp (yytext, "-")) || (!strcmp (yytext, "<")) ||
	(!strcmp (yytext, ">")) || (!strcmp (yytext, "<>")) ||
	(!strcmp (yytext, "<=")) || (!strcmp (yytext, ">=")) ||
	(!strcmp (yytext, "@")))
    printf ("$%s$", yytext);
  else if (!strcmp (yytext, "^"))
    printf ("\\%s{}", yytext);
  else if (!strcmp (yytext, "@"))
    printf ("\\%s", yytext);
  else
    echo ();
  IpPos += yyleng;
  OpPos += yyleng;
}

main (int argc, char *argv[]) {
  if (argc > 1)
    if (argc == 2)
      if (!strcmp (argv[1], "-n"))
        LineNum = 1;				/* turn on line numbering */
      else
        printf ("%s: unknown option %s\n", Prog, argv[1]);
    else
      printf ("%s: too many arguments\n", Prog);
  yylex ();
}

%}

%a 15000					/* increase number of transitions */

%e 2500						/* increase number of nodes */

%n 1500						/* increase number of states */

%o 15000					/* increase number of output slots */

%p 10000					/* increase number of positions */

Letter		[a-zA-Z_]
Digit		[0-9]

Identifier      {Letter}({Letter}|{Digit})*

Comment		"\/*"|"*\/"

Symbol1		"+"|"-"|"*"|"/"|"="|"<"|">"|"["|"]"|"."|","|":"|";"
Symbol2		"^"|"("|")"|"!"
Symbol		{Symbol1}|{Symbol2}

String          "'"

%%

^" "+ |
  " "" "+	IpPos += yyleng;

\t+ {
  IpPos = Tab * (IpPos / Tab + yyleng);
}

\n {
  IpPos = 0;
  OpPos = 0;
  TabPos = 0;
  if (LineNum > 0)
    printf ("\\`\\footnotesize%d\\\\\n", LineNum++);
  else
    printf ("\\\\\n");
}

{Comment} {
  Input ();
  InComment = !InComment;
  OutputId ();
}

{String} {
  Input ();
  if (!InComment) InString = !InString;
  OutputId ();
}

{Symbol} {
  Input ();
  OutputSym ();
}

(active|adding|all|alternative|and|any|as|atleast|axioms)	|
(Active|Adding|All|Alternative|And|Any|As|AtLeast|Axioms)	|
(block|call|channel|comment|connect|connection|constant)	|
(Block|Call|Channel|Comment|Connect|Connection|Constant)	|
(constants|create|dcl|decision|default|else|end|endalternative)	|
(Constants|Create|Dcl|Decision|Default|Else|End|EndAlternative)	|
(endblock|endchannel|endconnection|enddecision|endgenerator)	|
(EndBlock|EndChannel|EndConnection|EndDecision|EndGenerator)	|
(endmacro|endnewtype|endoperator|endpackage|endprocedure)	|
(EndMacro|EndNewType|EndOperator|EndPackage|EndProcedure)	|
(endprocess|endrefinement|endselect|endservice|endstate)	|
(EndProcess|EndRefinement|EndSelect|EndService|EndState)	|
(endsubstructure|endsyntype|endsystem|env|error|export)		|
(EndSubStructure|EndSynType|EndSystem|Env|Error|Export)		|
(exported|external|fi|finalized|for|fpar|from|gate|generator)	|
(Exported|External|Fi|Finalized|For|FPar|From|Gate|Generator)	|
(if|import|imported|in|inherits|input|interface|join|literal)	|
(If|Import|Imported|In|Inherits|Input|Interface|Join|Literal)	|
(literals|macro|macrodefinition|macroid|map|mod|nameclass)	|
(Literals|Macro|MacroDefinition|MacroId|Map|Mod|NameClass)	|
(newtype|nextstate|nodelay|noequality|none|not|now|offspring)	|
(NewType|NextState|NoDelay|NoEquality|None|Not|Now|Offspring)	|
(operator|operators|or|ordering|out|output|package|parent)	|
(Operator|Operators|Or|Ordering|Out|Output|Package|Parent)	|
(priority|procedure|process|provided|redefined|referenced)	|
(Priority|Procedure|Process|Provided|Redefined|Referenced)	|
(refinement|rem|remote|reset|return|returns|revealed|reverse)	|
(Refinement|Rem|Remote|Reset|Return|Returns|Revealed|Reverse)	|
(route|save|select|self|sender|service|set|signal|signallist)	|
(Route|Save|Select|Self|Sender|Service|Set|Signal|SignalList)	|
(signalroute|signalset|spelling|start|state|stop|struct)	|
(SignalRoute|SignalSet|Spelling|Start|State|Stop|Struct)	|
(substructure|synonym|syntype|system|task|then|this|timer|to)	|
(SubStructure|Synonym|SynType|System|Task|Then|This|Timer|To)	|
(type|use|via|view|viewed|virtual|with|xor)			|
(Type|Use|Via|View|Viewed|Virtual|With|Xor) {
  Input ();
  if (!InComment && !InString)
    printf ("{\\bf ");
  OutputId ();
  if (!InComment && !InString)
    printf ("}");
}

{Identifier}|. {
  Input ();
  OutputId ();
}

%%

/* output current text */

void echo () {
  ECHO;
}

