package uk.ac.stir.cs.terrawiz;

import java.awt.*;
import java.awt.event.*;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Properties;

import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.*;

import org.apache.xerces.dom.DOMImplementationImpl;
import org.apache.xerces.parsers.DOMParser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.SAXNotRecognizedException;
import org.xml.sax.SAXNotSupportedException;
import org.xml.sax.SAXParseException;
import org.xml.sax.helpers.DefaultHandler;

/**
  Class to deal with advanced mode.

  @author	Tao (Fred) Lu, Kenneth J. Turner
  @version	1.0 (6th April 2009)
*/

class ErrorChecker extends DefaultHandler {

  /**
    Report parse error.

    @param e	parsing exception
  */

  public void error(SAXParseException e) {
    JOptionPane.showMessageDialog(null, "The file is invalid due to a " +
      "parsing problem:  " + e.getMessage());
    System.exit(1);
  }

  /**
    Report parse warning.

    @param e	parsing exception
  */

  public void warning(SAXParseException e) {
    JOptionPane.showMessageDialog(null, "The file is invalid due to a " +
      "parsing problem:  " + e.getMessage());
    System.exit(1);
  }

  /**
    Report parse fatal error.

    @param e  Parameter
  */

  public void fatalError(SAXParseException e) {
    JOptionPane.showMessageDialog(null, "The file is invalid due to a " +
      "parsing problem:  " + e.getMessage());
    System.exit(1);
  }

}

/**
  Class to deal with advanced settings.

  @author	Tao (Fred) Lu, Kenneth J. Turner
  @version	1.0 (6th April 2009)
*/

public class Advanced extends JFrame implements ActionListener {

  // menus
  private JMenuBar menubar;
  private JMenu fileMenu, helpMenu, openRecentMenu;
  private JMenuItem newFile, openFile, save, saveAs, reload, exit,
    helpContent, about,
    recentFile1, recentFile2, recentFile3, recentFile4,
    preference;

  // tabbed pane
  private JTabbedPane switchable;

  // panels
  private JPanel panelLake, panelMountain, panelSurface, panelCloud,
    panelLight, panelCamera;

  // labels
  private JLabel labEnableLake, labEnableMount, labEnableSurface,
    labEnableCloud;

  // checkboxes (enable or disable certain thing such as lake, mountain)
  private JCheckBox cboxLake, cboxMountain, cboxSurface, cboxCloud;

  // lake panel
  private JRadioButton radioLakeDepth1, radioLakeDepth2,
    radioLakeRadius1, radioLakeRadius2,
    radioLakeRough1, radioLakeRough2,
    radioLakeWave1, radioLakeWave2,
    radioLakeReflect1, radioLakeReflect2;

  // mountain panel
  private JRadioButton radioMountRough1, radioMountRough2,
    radioMountFlat1, radioMountFlat2;

  // surface panel
  private JRadioButton radioSurface1, radioSurface2,
    radioSurfaceMaxalt1, radioSurfaceMaxalt2, radioSurfaceMaxalt3,
    radioSurfaceMinalt1, radioSurfaceMinalt2, radioSurfaceMinalt3;

  // cloud panel
  private JRadioButton radioCloudCategory1, radioCloudCategory2,
    radioCloudColor1, radioCloudColor2,
    radioCloudDensity1, radioCloudDensity2,
    radioCloudAltitude1, radioCloudAltitude2,
    radioCloudDepth1, radioCloudDepth2;

  // lighting panel
  private JLabel labLightHeading, labSunHeight;

  private JRadioButton radioLightHeading1, radioLightHeading2,
    radioLightHeading3, radioLightHeading4,
    radioLightElevation1, radioLightElevation2,
    radioLightElevation3;

  // camera panel

  private JLabel labCamHeight, labCamHeading, labSurfaceMax, labSurfaceMin;
  private JRadioButton radioCameraHeight1, radioCameraHeight2,
    radioCameraHeading1, radioCameraHeading2,
    radioCameraHeading3, radioCameraHeading4;

  // all panels share this buttonGroup object
  private ButtonGroup aButtonGroup;

  // objects for parsering DOM

  private DOMParser parser;
  private Document domTGD, domTGW, domSetting;

  // elements for TGD tree
  private Element rootTGD, lakeTGD, waterShader, fractalShader,
    surfaceShader, surfaceLayer,
    cloudLayer, cloudShader;

  // elements for TGW tree
  private Element rootTGW, lakeTGW, mountain, surface, cloud, lighting, camera;

  // elements for recent files tree
  private Element rootSetting, recentFile, language, mode;
  private Node file;
  private Text fileTextNode;

  // GUIs for file choosing
  private JFileChooser fileChooser;
  private File selectedFile;

  // settings
  private int[][] loadedSetting;
  private int[][] tempSetting;

  // property for translations (between English and Chinese)
  private Properties translation;

  // strings
  private String settingsFullPath;
  private String homeDirectory = System.getProperty("user.home");

  public String lang;

  /** Constructor of Advanced object.  */

  public Advanced() {
    initWindow();				// set window size and location
    initLang();					// initalize language setting
    createGUI();
    initParser();
    initVariables();
    if (!loadSettingDOM(Common.settingsFile)) // if no setting file
      initSettingFile();
    resetLang();
  }

  /**
    Constructor of Advanced object for language.

    @param lang  user language
  */

  public Advanced(String lang) {
    initWindow();				// Set window size and location
    if (!loadSettingDOM(Common.settingsFile)) // if no setting file
      initSettingFile();
    initLang(lang);				// initalize language setting
    createGUI();
    initParser();
    initVariables();
    resetLang();
  }

  /**
    Get input stream for resource/library file.

    @param fileName	file name relative to program base directory
    @return		file input stream
  */

  private InputStream getResourceStream(String fileName) {
    return(getClass().getClassLoader().getResourceAsStream(fileName));
  }

  /** Change language of menus, title and buttons. */

  void resetLang() {
    fileMenu.setText(translate("menu.file"));
    helpMenu.setText(translate("menu.help"));

    newFile.setText(translate("menuitem.new"));
    openFile.setText(translate("menuitem.open"));
    openRecentMenu.setText(translate("menuitem.openrecent"));
    save.setText(translate("menuitem.save"));
    saveAs.setText(translate("menuitem.saveas"));
    reload.setText(translate("menuitem.reload"));
    exit.setText(translate("menuitem.exit"));

    preference.setText(translate("menuitem.preference"));

    helpContent.setText(translate("menuitem.helpcontents"));
    about.setText(translate("menuitem.about"));

    switchable.setTitleAt(0, translate("tab.lake"));
    switchable.setTitleAt(1, translate("tab.mountain"));
    switchable.setTitleAt(2, translate("tab.surface"));
    switchable.setTitleAt(3, translate("tab.cloud"));
    switchable.setTitleAt(4, translate("tab.lighting"));
    switchable.setTitleAt(5, translate("tab.camera"));

    labEnableLake.setText(translate("label.lake"));
    radioLakeDepth1.setText(translate("label.lake.shallow"));
    radioLakeDepth2.setText(translate("label.lake.deep"));
    radioLakeRadius1.setText(translate("label.lake.narrow"));
    radioLakeRadius2.setText(translate("label.lake.broad"));
    radioLakeRough1.setText(translate("label.lake.smooth"));
    radioLakeRough2.setText(translate("label.lake.rough"));
    radioLakeWave1.setText(translate("label.lake.smallwave"));
    radioLakeWave2.setText(translate("label.lake.bigwave"));
    radioLakeReflect1.setText(translate("label.lake.lowreflect"));
    radioLakeReflect2.setText(translate("label.lake.highreflect"));

    labEnableMount.setText(translate("label.mountain"));
    radioMountRough1.setText(translate("label.mountain.smooth"));
    radioMountRough2.setText(translate("label.mountain.rough"));
    radioMountFlat1.setText(translate("label.mountain.flat"));
    radioMountFlat2.setText(translate("label.mountain.bumpy"));

    labEnableSurface.setText(translate("label.surface"));
    radioSurface1.setText(translate("label.surface.snow"));
    radioSurface2.setText(translate("label.surface.grass"));
    labSurfaceMax.setText(translate("label.surface.labmax"));
    labSurfaceMin.setText(translate("label.surface.labmin"));
    radioSurfaceMaxalt1.setText(translate("label.surface.lowmaxalt"));
    radioSurfaceMaxalt2.setText(translate("label.surface.highmaxalt"));
    radioSurfaceMaxalt3.setText(translate("label.surface.nonemaxalt"));
    radioSurfaceMinalt1.setText(translate("label.surface.lowminalt"));
    radioSurfaceMinalt2.setText(translate("label.surface.highminalt"));
    radioSurfaceMinalt3.setText(translate("label.surface.noneminalt"));

    labEnableCloud.setText(translate("label.cloud"));
    radioCloudCategory1.setText(translate("label.cloud.alto"));
    radioCloudCategory2.setText(translate("label.cloud.cirrus"));
    radioCloudColor1.setText(translate("label.cloud.white"));
    radioCloudColor2.setText(translate("label.cloud.gray"));
    radioCloudAltitude1.setText(translate("label.cloud.lowalt"));
    radioCloudAltitude2.setText(translate("label.cloud.highalt"));
    radioCloudDepth1.setText(translate("label.cloud.thin"));
    radioCloudDepth2.setText(translate("label.cloud.thick"));
    radioCloudDensity1.setText(translate("label.cloud.lowdensity"));
    radioCloudDensity2.setText(translate("label.cloud.highdensity"));

    labLightHeading.setText(translate("label.lighting.lightheading"));
    radioLightHeading1.setText(translate("label.lighting.east"));
    radioLightHeading2.setText(translate("label.lighting.north"));
    radioLightHeading3.setText(translate("label.lighting.west"));
    radioLightHeading4.setText(translate("label.lighting.south"));
    labSunHeight.setText(translate("label.lighting.sunheight"));
    radioLightElevation1.setText(translate("label.lighting.low"));
    radioLightElevation2.setText(translate("label.lighting.medium"));
    radioLightElevation3.setText(translate("label.lighting.high"));

    labCamHeight.setText(translate("label.camera.height"));
    radioCameraHeight1.setText(translate("label.camera.low"));
    radioCameraHeight2.setText(translate("label.camera.high"));
    labCamHeading.setText(translate("label.camera.heading"));
    radioCameraHeading1.setText(translate("label.camera.east"));
    radioCameraHeading2.setText(translate("label.camera.north"));
    radioCameraHeading3.setText(translate("label.camera.west"));
    radioCameraHeading4.setText(translate("label.camera.south"));
  }

  /** Initialize the window. */

  private void initWindow() {
    setTitle(translate("title.advancedmodenew"));
    setSize(Common.frameWidth, Common.frameHeight);
    int w = (Toolkit.getDefaultToolkit().getScreenSize().width -
      Common.tabWidth) / 2;
    int h = (Toolkit.getDefaultToolkit().getScreenSize().height -
      Common.tabHeight) / 2;
    setLocation(w, h);
    setVisible(true);
  }

  /** Initialize the parser, maintaining two DOM trees (TGD, TGW). */

  private void initParser() {
    loadTGDDOM(Common.terragenTemplate);
    createTGWDOM();
  }

  /**
    Load TGD DOM tree from a standard file.

    @param fileName	TGD file name
  */

  private void loadTGDDOM(String fileName) {
    parser = new DOMParser();
    try {
      parser.parse(new InputSource(getResourceStream(fileName)));
    }
    catch (SAXException e1) {
      e1.printStackTrace();
    }
    catch (IOException e1) {			// file doesn't exist
      JOptionPane.showMessageDialog(this, translate("err.filenotexist") +
	" - " + fileName);
      setWindowTitle("");
      Common.fileState = Common.createdFile;
      return;
    }

    domTGD = parser.getDocument();
    rootTGD = domTGD.getDocumentElement();
  }

  /** Create TGW DOM tree from scratch. */

  private void createTGWDOM() {
    domTGW = DOMImplementationImpl.getDOMImplementation().createDocument(
      null, null, null);
    rootTGW = domTGW.createElement("terragen");
    rootTGW.setAttribute(
      "xmlns:xsi", "http://www.w3.org/2001/XMLSchema-instance");
    rootTGW.setAttribute(
      "xsi:noNamespaceSchemaLocation", Common.wizardURL);
    lighting = domTGW.createElement("lighting");
    camera = domTGW.createElement("camera");
    lighting.setAttribute("heading", Common.tgwLightingHeading1);
    lighting.setAttribute("elevation", Common.tgwLightingHeading1);
    camera.setAttribute("height", Common.tgwCameraView1);
    camera.setAttribute("heading", Common.tgwCameraHeading1);

    rootTGW.appendChild(lighting);
    rootTGW.appendChild(camera);
    domTGW.appendChild(rootTGW);
  }

  /**
    Load TGW DOM tree from a standard file.

    @param fileName	TGW file name
  */

  private void loadTGWDOM(String fileName) {
    parser = new DOMParser();
    try {
      parser.setFeature("http://xml.org/sax/features/validation", true);
      parser.setFeature(
	"http://apache.org/xml/features/validation/schema", true);
    }
    catch (SAXNotRecognizedException e) {
      e.printStackTrace();
    }
    catch (SAXNotSupportedException e) {
      e.printStackTrace();
    }
    ErrorChecker errors = new ErrorChecker();
    parser.setErrorHandler(errors);

    try {
      parser.parse(fileName);
    }
    catch (SAXException e1) {
      e1.printStackTrace();
    }
    catch (IOException e1) {
      // File not found
      JOptionPane.showMessageDialog(this, translate("err.filenotexist") +
	" - " + fileName);
      setWindowTitle("");
      Common.fileState = Common.createdFile;
      return;
    }

    domTGW = parser.getDocument();
    rootTGW = domTGW.getDocumentElement();
  }

  /** Initialize variables. */

  private void initVariables() {
    loadedSetting = new int[6][6];
    for (int i = 0; i < loadedSetting.length; i++)
      for (int j = 0; j < loadedSetting[i].length; j++)
	loadedSetting[i][j] = 0;

    tempSetting = new int[6][6];
    for (int i = 0; i < tempSetting.length; i++)
      for (int j = 0; j < tempSetting[i].length; j++)
	tempSetting[i][j] = 0;

    loadedSetting[4][0] = 1;			// lighting panel is always on
    loadedSetting[5][0] = 1;			// camera panel is always on
  }

  /** Initialize settings files. */

  private void initSettingFile() {
    domSetting =
      DOMImplementationImpl.getDOMImplementation().createDocument(
	null, null, null);
    rootSetting = domSetting.createElement("Setting");
    recentFile = domSetting.createElement("RecentFile");

    language = domSetting.createElement("Language");
    language.appendChild(domSetting.createTextNode("en-GB"));
    mode = domSetting.createElement("Mode");
    mode.appendChild(domSetting.createTextNode("Advanced"));

    rootSetting.appendChild(recentFile);
    rootSetting.appendChild(language);
    rootSetting.appendChild(mode);

    domSetting.appendChild(rootSetting);

    // create settings file
    settingsFullPath = homeDirectory + Common.settingsFile;
    Common.writeXML(domSetting, settingsFullPath);
  }

  /**
    Load recent DOM tree from a file.

    @param fileName	tree file name
    @return		true if loading successful
  */

  private boolean loadSettingDOM(String fileName) {
    parser = new DOMParser();
    try {
      settingsFullPath = homeDirectory + fileName;
      parser.parse(settingsFullPath);
    }
    catch (SAXException e1) {
      return false;
    }
    catch (IOException e1) {
      return false;
    }

    domSetting = parser.getDocument();
    rootSetting = domSetting.getDocumentElement();
    // look for node RecentFile
    NodeList nodeList1 = rootSetting.getElementsByTagName("RecentFile");
    if (nodeList1.getLength() == 0)
      return false;
    else
      recentFile = (Element) nodeList1.item(0);

    changeRecentFiles();
    return true;
  }

  /** Set recent files in menu based on DOM. */

  private void changeRecentFiles() {
    NodeList nodeList1 = rootSetting.getElementsByTagName("File");
    if (nodeList1.getLength() == 0) {
      recentFile1.setVisible(false);
      recentFile2.setVisible(false);
      recentFile3.setVisible(false);
      recentFile4.setVisible(false);
    }
    else {
      Common.openedFileNum = nodeList1.getLength();
      for (int i = 0; i < Common.openedFileNum; i++) {
	Node aNode = nodeList1.item(i);
	if (aNode.getFirstChild().getNodeValue() != null)
	  switch (i) {
	      case 0:
		recentFile1.setText(aNode.getFirstChild().getNodeValue());
		recentFile1.setVisible(true);
		break;
	      case 1:
		recentFile2.setText(aNode.getFirstChild().getNodeValue());
		recentFile2.setVisible(true);
		break;
	      case 2:
		recentFile3.setText(aNode.getFirstChild().getNodeValue());
		recentFile3.setVisible(true);
		break;
	      case 3:
		recentFile4.setText(aNode.getFirstChild().getNodeValue());
		recentFile4.setVisible(true);
		break;
	      default:
		break;
	  }
      }
    }
  }

  /** Initialize language.  */

  private void initLang() {
    parser = new DOMParser();
    try {
      settingsFullPath = homeDirectory + Common.settingsFile;
      parser.parse(settingsFullPath);
    }
    catch (SAXException e1) {
    }
    catch (IOException e1) {
    }

    domSetting = parser.getDocument();
    rootSetting = domSetting.getDocumentElement();
    NodeList nodeList1 = rootSetting.getElementsByTagName("Language");
    lang = nodeList1.item(0).getFirstChild().getNodeValue();
    translation = new Properties();
    InputStream languageFile = getResourceStream("res/" + lang + ".properties");
    try {
      InputStreamReader languageInput =
	new InputStreamReader(languageFile, "UTF-16");
      translation.load(languageInput);
      languageInput.close();
    }
    catch (IOException exception) {
      System.err.println("Wrong path or filename");
    }
    setTitle(translate("title.advancedmodenew"));
  }

  /**
    Initialize language with specific value.

    @param lang		user language
  */

  public void initLang(String lang) {
    translation = new Properties();
    InputStream languageFile = getResourceStream("res/" + lang + ".properties");
    try {
      InputStreamReader languageInput =
	new InputStreamReader(languageFile, "UTF-16");
      translation.load(languageInput);
      languageInput.close();
    }
    catch (IOException exception) {
      System.err.println("Wrong path or filename");
    }
    changeSetting(lang, "Language");
  }

  /**
    Change the setting of a certain element.

    @param value	value to change to
    @param nodeName	node to change
  */

  public void changeSetting(String value, String nodeName) {
    Common.changeElem(rootSetting, nodeName, value);
    Common.writeXML(domSetting, homeDirectory + Common.settingsFile);
  }

  /**
    Translate language term.

    @param word  string from en-GB or zh-CN properties, such as "menu.file"
    @return      translated string
  */

  public String translate(String word) {
    String translated = "";
    if (translation == null ||
      (translated = translation.getProperty(word)) == null)
      translated = word + "?";
    return (translated);
  }

  /** Create GUI. */

  private void createGUI() {
    setDefaultCloseOperation(EXIT_ON_CLOSE);
    Container window = getContentPane();
    window.setLayout(new FlowLayout());

    // set icon
    ImageIcon titleIcon = new ImageIcon(Common.iconPath);
    setIconImage(titleIcon.getImage());

    // menu bar
    menubar = new JMenuBar();

    // menus
    fileMenu = new JMenu(translate("menu.file"));
    fileMenu.setMnemonic(KeyEvent.VK_F);		// set shortcut
    helpMenu = new JMenu(translate("menu.help"));
    helpMenu.setMnemonic(KeyEvent.VK_H);		// set shortcut

    // menu items

    // items in "File"
    newFile = new JMenuItem(translate("menuitem.new"));
    newFile.setIcon(new ImageIcon("res/new.png"));
    newFile.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_N, InputEvent.CTRL_MASK));
    openFile = new JMenuItem(translate("menuitem.open"));
    openFile.setIcon(new ImageIcon("res/open.png"));
    openFile.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_O, InputEvent.CTRL_MASK));
    openRecentMenu = new JMenu(translate("menuitem.openrecent"));
    openRecentMenu.setIcon(new ImageIcon("res/openrecent.png"));
    recentFile1 = new JMenuItem("");
    recentFile1.setVisible(false);
    recentFile2 = new JMenuItem("");
    recentFile2.setVisible(false);
    recentFile3 = new JMenuItem("");
    recentFile3.setVisible(false);
    recentFile4 = new JMenuItem("");
    recentFile4.setVisible(false);
    save = new JMenuItem(translate("menuitem.save"));
    save.setIcon(new ImageIcon("res/save.png"));
    save.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_S, InputEvent.CTRL_MASK));
    saveAs = new JMenuItem(translate("menuitem.saveas"));
    saveAs.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_S, InputEvent.CTRL_MASK + InputEvent.SHIFT_MASK));
    reload = new JMenuItem(translate("menuitem.reload"));
    reload.setIcon(new ImageIcon("res/reload.png"));
    reload.setEnabled(false);
    reload.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_R, InputEvent.CTRL_MASK));
    preference = new JMenuItem(translate("menuitem.preference"));
    preference.setIcon(new ImageIcon("res/preference.png"));
    preference.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_P, InputEvent.CTRL_MASK));
    exit = new JMenuItem(translate("menuitem.exit"));
    exit.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_ESCAPE, 0));

    // items in "Help"
    helpContent = new JMenuItem(translate("menuitem.helpcontents"));
    helpContent.setIcon(new ImageIcon("res/help.png"));
    helpContent.setAccelerator(KeyStroke.getKeyStroke(
      KeyEvent.VK_F1, 0));
    about = new JMenuItem(translate("menuitem.about"));

    // add into hierachical stucture
    fileMenu.add(newFile);
    fileMenu.addSeparator();
    fileMenu.add(openFile);
    fileMenu.add(openRecentMenu);
    fileMenu.addSeparator();
    fileMenu.add(save);
    fileMenu.add(saveAs);
    fileMenu.add(reload);
    fileMenu.addSeparator();
    fileMenu.add(preference);
    fileMenu.addSeparator();
    fileMenu.add(exit);

    openRecentMenu.add(recentFile1);
    openRecentMenu.add(recentFile2);
    openRecentMenu.add(recentFile3);
    openRecentMenu.add(recentFile4);

    helpMenu.add(helpContent);
    helpMenu.addSeparator();
    helpMenu.add(about);

    menubar.add(fileMenu);
    menubar.add(helpMenu);

    // add listeners
    newFile.addActionListener(this);
    openFile.addActionListener(this);
    recentFile1.addActionListener(this);
    recentFile2.addActionListener(this);
    recentFile3.addActionListener(this);
    recentFile4.addActionListener(this);
    save.addActionListener(this);
    saveAs.addActionListener(this);
    exit.addActionListener(this);
    reload.addActionListener(this);
    helpContent.addActionListener(this);
    about.addActionListener(this);
    preference.addActionListener(this);

    // set menu bar to window
    setJMenuBar(menubar);

    //// lake panel ////

    panelLake = new JPanel();
    panelLake.setPreferredSize(new Dimension(
      Common.tabWidth, Common.tabHeight));
    panelLake.setLayout(new GridBagLayout());
    GridBagConstraints constraints1 = new GridBagConstraints();

    // lake label
    constraints1.gridx = 0;
    constraints1.gridy = 0;
    constraints1.anchor = GridBagConstraints.WEST;
    labEnableLake = new JLabel(translate("label.lake"));
    panelLake.add(labEnableLake, constraints1);

    // lake checkbox
    constraints1.gridx = 1;
    constraints1.gridy = 0;
    cboxLake = new JCheckBox();
    cboxLake.addActionListener(this);
    panelLake.add(cboxLake, constraints1);

    // option row

    // lake depth button group
    radioLakeDepth1 = new JRadioButton(translate("label.lake.shallow"), true);
    radioLakeDepth2 = new JRadioButton(translate("label.lake.deep"));
    radioLakeDepth1.addActionListener(this);
    radioLakeDepth2.addActionListener(this);
    radioLakeDepth1.setEnabled(false);
    radioLakeDepth2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLakeDepth1);
    aButtonGroup.add(radioLakeDepth2);

    // lake depth radio button 1
    constraints1.gridx = 3;
    constraints1.gridy = 1;
    panelLake.add(radioLakeDepth1, constraints1);

    // lake depth radio button 2
    constraints1.gridx = 4;
    constraints1.gridy = 1;
    panelLake.add(radioLakeDepth2, constraints1);

    // lake radius buttong roup
    radioLakeRadius1 = new JRadioButton(translate("label.lake.narrow"), true);
    radioLakeRadius2 = new JRadioButton(translate("label.lake.broad"));
    radioLakeRadius1.addActionListener(this);
    radioLakeRadius2.addActionListener(this);
    radioLakeRadius1.setEnabled(false);
    radioLakeRadius2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLakeRadius1);
    aButtonGroup.add(radioLakeRadius2);

    // lake radius radio button 1
    constraints1.gridx = 3;
    constraints1.gridy = 2;
    panelLake.add(radioLakeRadius1, constraints1);

    // lake radiusradio button 2
    constraints1.gridx = 4;
    constraints1.gridy = 2;
    panelLake.add(radioLakeRadius2, constraints1);

    // lake roughness button group
    radioLakeRough1 = new JRadioButton(translate("label.lake.smooth"), true);
    radioLakeRough2 = new JRadioButton(translate("label.lake.rough"));
    radioLakeRough1.addActionListener(this);
    radioLakeRough2.addActionListener(this);
    radioLakeRough1.setEnabled(false);
    radioLakeRough2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLakeRough1);
    aButtonGroup.add(radioLakeRough2);

    // lake roughness radio button 1
    constraints1.gridx = 3;
    constraints1.gridy = 3;
    panelLake.add(radioLakeRough1, constraints1);

    // lake roughness radio button 2
    constraints1.gridx = 4;
    constraints1.gridy = 3;
    panelLake.add(radioLakeRough2, constraints1);

    // lake wave button group
    radioLakeWave1 = new JRadioButton(translate("label.lake.smallwave"), true);
    radioLakeWave2 = new JRadioButton(translate("label.lake.bigwave"));
    radioLakeWave1.addActionListener(this);
    radioLakeWave2.addActionListener(this);
    radioLakeWave1.setEnabled(false);
    radioLakeWave2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLakeWave1);
    aButtonGroup.add(radioLakeWave2);

    // lake wave radio button 1
    constraints1.gridx = 3;
    constraints1.gridy = 4;
    panelLake.add(radioLakeWave1, constraints1);

    // lake wave radio button 2
    constraints1.gridx = 4;
    constraints1.gridy = 4;
    panelLake.add(radioLakeWave2, constraints1);

    // lake reflection button group
    radioLakeReflect1 =
      new JRadioButton(translate("label.lake.lowreflect"), true);
    radioLakeReflect2 = new JRadioButton(translate("label.lake.highreflect"));
    radioLakeReflect1.addActionListener(this);
    radioLakeReflect2.addActionListener(this);
    radioLakeReflect1.setEnabled(false);
    radioLakeReflect2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLakeReflect1);
    aButtonGroup.add(radioLakeReflect2);

    // lake reflection radio button 1
    constraints1.gridx = 3;
    constraints1.gridy = 5;
    panelLake.add(radioLakeReflect1, constraints1);

    // lake reflection radio button 2
    constraints1.gridx = 4;
    constraints1.gridy = 5;
    panelLake.add(radioLakeReflect2, constraints1);


    //// mountain panel ////

    panelMountain = new JPanel();
    panelMountain.setPreferredSize(new Dimension(
      Common.tabWidth, Common.tabHeight));
    panelMountain.setLayout(new GridBagLayout());
    GridBagConstraints constraints2 = new GridBagConstraints();

    // mountain label
    constraints2.gridx = 0;
    constraints2.gridy = 0;
    constraints2.anchor = GridBagConstraints.WEST;
    labEnableMount = new JLabel(translate("label.mountain"));
    panelMountain.add(labEnableMount, constraints2);

    // mountain checkbox
    constraints2.gridx = 1;
    constraints2.gridy = 0;
    cboxMountain = new JCheckBox();
    cboxMountain.addActionListener(this);
    panelMountain.add(cboxMountain, constraints2);

    // mountain roughness button group
    radioMountRough1 =
      new JRadioButton(translate("label.mountain.smooth"), true);
    radioMountRough2 =
      new JRadioButton(translate("label.mountain.rough"));
    radioMountRough1.addActionListener(this);
    radioMountRough2.addActionListener(this);
    radioMountRough1.setEnabled(false);
    radioMountRough2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioMountRough1);
    aButtonGroup.add(radioMountRough2);

    // mountain radio button 1
    constraints2.gridx = 3;
    constraints2.gridy = 1;
    panelMountain.add(radioMountRough1, constraints2);

    // mountain radio button 2
    constraints2.gridx = 4;
    constraints2.gridy = 1;
    panelMountain.add(radioMountRough2, constraints2);

    // option row
    // mountain roughness button group
    radioMountFlat1 = new JRadioButton(translate("label.mountain.flat"), true);
    radioMountFlat2 = new JRadioButton(translate("label.mountain.bumpy"));
    radioMountFlat1.addActionListener(this);
    radioMountFlat2.addActionListener(this);
    radioMountFlat1.setEnabled(false);
    radioMountFlat2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioMountFlat1);
    aButtonGroup.add(radioMountFlat2);

    // mountain radio button 1
    constraints2.gridx = 3;
    constraints2.gridy = 2;
    panelMountain.add(radioMountFlat1, constraints2);

    // mountain radio button 2
    constraints2.gridx = 4;
    constraints2.gridy = 2;
    panelMountain.add(radioMountFlat2, constraints2);


    //// surface panel ////

    panelSurface = new JPanel();
    panelSurface.setPreferredSize(new Dimension(
      Common.tabWidth, Common.tabHeight));
    panelSurface.setLayout(new GridBagLayout());
    GridBagConstraints constraints3 = new GridBagConstraints();

    // surface label
    constraints3.gridx = 0;
    constraints3.gridy = 0;
    constraints3.anchor = GridBagConstraints.WEST;
    labEnableSurface = new JLabel(translate("label.surface"));
    panelSurface.add(labEnableSurface, constraints3);

    // surface checkbox
    constraints3.gridx = 1;
    constraints3.gridy = 0;
    cboxSurface = new JCheckBox();
    cboxSurface.addActionListener(this);
    panelSurface.add(cboxSurface, constraints3);

    // surface skin button group
    radioSurface1 = new JRadioButton(translate("label.surface.snow"), true);
    radioSurface2 = new JRadioButton(translate("label.surface.grass"));
    radioSurface1.addActionListener(this);
    radioSurface2.addActionListener(this);
    radioSurface1.setEnabled(false);
    radioSurface2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioSurface1);
    aButtonGroup.add(radioSurface2);

    // surface skin radio button 1
    constraints3.gridx = 3;
    constraints3.gridy = 1;
    panelSurface.add(radioSurface1, constraints3);

    // surface skin radio button 2
    constraints3.gridx = 4;
    constraints3.gridy = 1;
    panelSurface.add(radioSurface2, constraints3);

    // surface max altitude button group
    radioSurfaceMaxalt1 =
      new JRadioButton(translate("label.surface.lowmaxalt"), true);
    radioSurfaceMaxalt2 =
      new JRadioButton(translate("label.surface.highmaxalt"));
    radioSurfaceMaxalt3 =
      new JRadioButton(translate("label.surface.nonemaxalt"));
    radioSurfaceMaxalt1.addActionListener(this);
    radioSurfaceMaxalt2.addActionListener(this);
    radioSurfaceMaxalt3.addActionListener(this);
    radioSurfaceMaxalt1.setEnabled(false);
    radioSurfaceMaxalt2.setEnabled(false);
    radioSurfaceMaxalt3.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioSurfaceMaxalt1);
    aButtonGroup.add(radioSurfaceMaxalt2);
    aButtonGroup.add(radioSurfaceMaxalt3);

    constraints3.gridx = 2;
    constraints3.gridy = 2;
    labSurfaceMax = new JLabel(translate("label.surface.labmax"));
    labSurfaceMax.setEnabled(false);
    panelSurface.add(labSurfaceMax, constraints3);

    // surface max altitude radio button 1
    constraints3.gridx = 3;
    constraints3.gridy = 2;
    panelSurface.add(radioSurfaceMaxalt1, constraints3);

    // surface max altitude radio button 2
    constraints3.gridx = 4;
    constraints3.gridy = 2;
    panelSurface.add(radioSurfaceMaxalt2, constraints3);

    // surface max altitude radio button 3
    constraints3.gridx = 5;
    constraints3.gridy = 2;
    panelSurface.add(radioSurfaceMaxalt3, constraints3);

    // surface min altitude button group
    radioSurfaceMinalt1 =
      new JRadioButton(translate("label.surface.lowminalt"), true);
    radioSurfaceMinalt2 =
      new JRadioButton(translate("label.surface.highminalt"));
    radioSurfaceMinalt3 =
      new JRadioButton(translate("label.surface.noneminalt"));
    radioSurfaceMinalt1.addActionListener(this);
    radioSurfaceMinalt2.addActionListener(this);
    radioSurfaceMinalt3.addActionListener(this);
    radioSurfaceMinalt1.setEnabled(false);
    radioSurfaceMinalt2.setEnabled(false);
    radioSurfaceMinalt3.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioSurfaceMinalt1);
    aButtonGroup.add(radioSurfaceMinalt2);
    aButtonGroup.add(radioSurfaceMinalt3);

    constraints3.gridx = 2;
    constraints3.gridy = 3;
    labSurfaceMin = new JLabel(translate("label.surface.labmin"));
    labSurfaceMin.setEnabled(false);
    panelSurface.add(labSurfaceMin, constraints3);

    // surface min altitude radio button 1
    constraints3.gridx = 3;
    constraints3.gridy = 3;
    panelSurface.add(radioSurfaceMinalt1, constraints3);

    // surface min altitude radio button 2
    constraints3.gridx = 4;
    constraints3.gridy = 3;
    panelSurface.add(radioSurfaceMinalt2, constraints3);

    // surface min altitude radio button 3
    constraints3.gridx = 5;
    constraints3.gridy = 3;
    panelSurface.add(radioSurfaceMinalt3, constraints3);


    //// cloud panel ////

    panelCloud = new JPanel();
    panelCloud.setPreferredSize(new Dimension(Common.tabWidth, Common.tabHeight));
    panelCloud.setLayout(new GridBagLayout());
    GridBagConstraints constraints4 = new GridBagConstraints();

    // cloud label
    constraints4.gridx = 0;
    constraints4.gridy = 0;
    constraints4.anchor = GridBagConstraints.WEST;
    labEnableCloud = new JLabel(translate("label.cloud"));
    panelCloud.add(labEnableCloud, constraints4);

    // cloud checkbox
    constraints4.gridx = 1;
    constraints4.gridy = 0;
    cboxCloud = new JCheckBox();
    cboxCloud.addActionListener(this);
    panelCloud.add(cboxCloud, constraints4);

    // cloud category button group
    radioCloudCategory1 = new JRadioButton(translate("label.cloud.alto"), true);
    radioCloudCategory2 = new JRadioButton(translate("label.cloud.cirrus"));
    radioCloudCategory1.addActionListener(this);
    radioCloudCategory2.addActionListener(this);
    radioCloudCategory1.setEnabled(false);
    radioCloudCategory2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCloudCategory1);
    aButtonGroup.add(radioCloudCategory2);

    // cloud category radio button 1
    constraints4.gridx = 3;
    constraints4.gridy = 1;
    panelCloud.add(radioCloudCategory1, constraints4);

    // cloud category radio button 2
    constraints4.gridx = 4;
    constraints4.gridy = 1;
    panelCloud.add(radioCloudCategory2, constraints4);

    // cloud colour button group
    radioCloudColor1 = new JRadioButton(translate("label.cloud.white"), true);
    radioCloudColor2 = new JRadioButton(translate("label.cloud.gray"));
    radioCloudColor1.addActionListener(this);
    radioCloudColor2.addActionListener(this);
    radioCloudColor1.setEnabled(false);
    radioCloudColor2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCloudColor1);
    aButtonGroup.add(radioCloudColor2);

    // cloud colour radio button 1
    constraints4.gridx = 3;
    constraints4.gridy = 2;
    panelCloud.add(radioCloudColor1, constraints4);

    // cloud colour radio button 2
    constraints4.gridx = 4;
    constraints4.gridy = 2;
    panelCloud.add(radioCloudColor2, constraints4);

    // cloud altitude button group
    radioCloudAltitude1 =
      new JRadioButton(translate("label.cloud.lowalt"), true);
    radioCloudAltitude2 =
      new JRadioButton(translate("label.cloud.highalt"));
    radioCloudAltitude1.addActionListener(this);
    radioCloudAltitude2.addActionListener(this);
    radioCloudAltitude1.setEnabled(false);
    radioCloudAltitude2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCloudAltitude1);
    aButtonGroup.add(radioCloudAltitude2);

    // cloud altitude radio button 1
    constraints4.gridx = 3;
    constraints4.gridy = 3;
    panelCloud.add(radioCloudAltitude1, constraints4);

    // cloud altitude radio button 2
    constraints4.gridx = 4;
    constraints4.gridy = 3;
    panelCloud.add(radioCloudAltitude2, constraints4);

    // cloud depth button group
    radioCloudDepth1 = new JRadioButton(translate("label.cloud.thin"), true);
    radioCloudDepth2 = new JRadioButton(translate("label.cloud.thick"));
    radioCloudDepth1.addActionListener(this);
    radioCloudDepth2.addActionListener(this);
    radioCloudDepth1.setEnabled(false);
    radioCloudDepth2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCloudDepth1);
    aButtonGroup.add(radioCloudDepth2);

    // cloud depth radio button 1
    constraints4.gridx = 3;
    constraints4.gridy = 4;
    panelCloud.add(radioCloudDepth1, constraints4);

    // cloud depth radio button 2
    constraints4.gridx = 4;
    constraints4.gridy = 4;
    panelCloud.add(radioCloudDepth2, constraints4);

    // cloud density button group
    radioCloudDensity1 =
      new JRadioButton(translate("label.cloud.lowdensity"), true);
    radioCloudDensity2 =
      new JRadioButton(translate("label.cloud.highdensity"));
    radioCloudDensity1.addActionListener(this);
    radioCloudDensity2.addActionListener(this);
    radioCloudDensity1.setEnabled(false);
    radioCloudDensity2.setEnabled(false);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCloudDensity1);
    aButtonGroup.add(radioCloudDensity2);

    // cloud density radio button 1
    constraints4.gridx = 3;
    constraints4.gridy = 5;
    panelCloud.add(radioCloudDensity1, constraints4);

    // cloud density radio button 2
    constraints4.gridx = 4;
    constraints4.gridy = 5;
    panelCloud.add(radioCloudDensity2, constraints4);


    //// light panel ////

    panelLight = new JPanel();
    panelLight.setPreferredSize(new Dimension(Common.tabWidth, Common.tabHeight));
    panelLight.setLayout(new GridBagLayout());
    GridBagConstraints constraints5 = new GridBagConstraints();

    constraints5.gridx = 0;
    constraints5.gridy = 0;
    constraints5.anchor = GridBagConstraints.WEST;

    // light heading button group
    radioLightHeading1 =
      new JRadioButton(translate("label.lighting.east"), true);
    radioLightHeading2 = new JRadioButton(translate("label.lighting.north"));
    radioLightHeading3 = new JRadioButton(translate("label.lighting.west"));
    radioLightHeading4 = new JRadioButton(translate("label.lighting.south"));
    radioLightHeading1.addActionListener(this);
    radioLightHeading2.addActionListener(this);
    radioLightHeading3.addActionListener(this);
    radioLightHeading4.addActionListener(this);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLightHeading1);
    aButtonGroup.add(radioLightHeading2);
    aButtonGroup.add(radioLightHeading3);
    aButtonGroup.add(radioLightHeading4);

    // light heading
    constraints5.gridx = 3;
    constraints5.gridy = 1;
    labLightHeading = new JLabel(translate("label.lighting.lightheading"));
    panelLight.add(labLightHeading, constraints5);

    // light heading radio button 1
    constraints5.gridx = 4;
    constraints5.gridy = 1;
    panelLight.add(radioLightHeading1, constraints5);

    // light heading radio button 2
    constraints5.gridx = 5;
    constraints5.gridy = 1;
    panelLight.add(radioLightHeading2, constraints5);

    // light heading radio button 3
    constraints5.gridx = 6;
    constraints5.gridy = 1;
    panelLight.add(radioLightHeading3, constraints5);

    // light heading radio button 4
    constraints5.gridx = 7;
    constraints5.gridy = 1;
    panelLight.add(radioLightHeading4, constraints5);

    // light elevation button group
    radioLightElevation1 =
      new JRadioButton(translate("label.lighting.low"), true);
    radioLightElevation2 = new JRadioButton(translate("label.lighting.medium"));
    radioLightElevation3 = new JRadioButton(translate("label.lighting.high"));
    radioLightElevation1.addActionListener(this);
    radioLightElevation2.addActionListener(this);
    radioLightElevation3.addActionListener(this);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioLightElevation1);
    aButtonGroup.add(radioLightElevation2);
    aButtonGroup.add(radioLightElevation3);

    // sun height
    constraints5.gridx = 3;
    constraints5.gridy = 2;
    labSunHeight = new JLabel(translate("label.lighting.sunheight"));
    panelLight.add(labSunHeight, constraints5);

    // light elevation radio button 1
    constraints5.gridx = 4;
    constraints5.gridy = 2;
    panelLight.add(radioLightElevation1, constraints5);

    // light elevation radio button 2
    constraints5.gridx = 5;
    constraints5.gridy = 2;
    panelLight.add(radioLightElevation2, constraints5);

    // light elevation radio button 3
    constraints5.gridx = 6;
    constraints5.gridy = 2;
    panelLight.add(radioLightElevation3, constraints5);


    //// camera panel ///

    panelCamera = new JPanel();
    panelCamera.setPreferredSize(new Dimension(400, 200));
    panelCamera.setLayout(new GridBagLayout());
    GridBagConstraints constraints6 = new GridBagConstraints();

    // camera label
    constraints6.gridx = 0;
    constraints6.gridy = 0;
    constraints5.anchor = GridBagConstraints.WEST;

    // camera button group
    radioCameraHeight1 = new JRadioButton(translate("label.camera.low"), true);
    radioCameraHeight2 = new JRadioButton(translate("label.camera.high"));
    radioCameraHeight1.addActionListener(this);
    radioCameraHeight2.addActionListener(this);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCameraHeight1);
    aButtonGroup.add(radioCameraHeight2);

    // camera view label
    constraints6.gridx = 3;
    constraints6.gridy = 1;
    labCamHeight = new JLabel(translate("label.camera.height"));
    panelCamera.add(labCamHeight, constraints6);

    // camera view radio button 1
    constraints6.gridx = 4;
    constraints6.gridy = 1;
    panelCamera.add(radioCameraHeight1, constraints6);

    // camera view radio button 2
    constraints6.gridx = 5;
    constraints6.gridy = 1;
    panelCamera.add(radioCameraHeight2, constraints6);

    // camera view button group
    radioCameraHeading1 =
      new JRadioButton(translate("label.camera.east"), true);
    radioCameraHeading2 = new JRadioButton(translate("label.camera.north"));
    radioCameraHeading3 = new JRadioButton(translate("label.camera.west"));
    radioCameraHeading4 = new JRadioButton(translate("label.camera.south"));
    radioCameraHeading1.addActionListener(this);
    radioCameraHeading2.addActionListener(this);
    radioCameraHeading3.addActionListener(this);
    radioCameraHeading4.addActionListener(this);
    aButtonGroup = new ButtonGroup();
    aButtonGroup.add(radioCameraHeading1);
    aButtonGroup.add(radioCameraHeading2);
    aButtonGroup.add(radioCameraHeading3);
    aButtonGroup.add(radioCameraHeading4);

    // camera heading label
    constraints6.gridx = 3;
    constraints6.gridy = 2;
    labCamHeading = new JLabel(translate("label.camera.heading"));
    panelCamera.add(labCamHeading, constraints6);

    // camera heading radio button 1
    constraints6.gridx = 4;
    constraints6.gridy = 2;
    panelCamera.add(radioCameraHeading1, constraints6);

    // camera heading radio button 2
    constraints6.gridx = 5;
    constraints6.gridy = 2;
    panelCamera.add(radioCameraHeading2, constraints6);

    // camera heading radio button 3
    constraints6.gridx = 6;
    constraints6.gridy = 2;
    panelCamera.add(radioCameraHeading3, constraints6);

    // camera heading radio button 4
    constraints6.gridx = 7;
    constraints6.gridy = 2;
    panelCamera.add(radioCameraHeading4, constraints6);


    //// tabbed pane ///

    switchable = new JTabbedPane();		// will have tabs at the top
    switchable.setPreferredSize(
      new Dimension(Common.tabWidth, Common.tabHeight));
    switchable.add(translate("tab.lake"), panelLake);
    switchable.add(translate("tab.mountain"), panelMountain);
    switchable.add(translate("tab.surface"), panelSurface);
    switchable.add(translate("tab.cloud"), panelCloud);
    switchable.add(translate("tab.lighting"), panelLight);
    switchable.add(translate("tab.camera"), panelCamera);
    window.add(switchable);

  }						// end of createGUI

  /**
    Respond to action event.

    @param e		action event
  */

  public void actionPerformed(ActionEvent e) {
    checkOptions();				// check all options
    storeSetting(tempSetting);			// store temporary settings
    checkSetting();				// check temp./loaded settings

    // when New is clicked
    if (e.getSource() == newFile) {
      if (Common.fileChanged) {		// current setting changed
	if (checkSave(1)) {			// prompt for save
	  Common.fileState = Common.createdFile;
	  Common.fileCategory = Common.tgwfile;
	  Common.fileChanged = false;
	  resetAllSetting();			// reset all the options in UI
	  setTitle(translate("title.advancedmodenew"));
	}
      }
      else {
	Common.fileState = Common.createdFile;
	Common.fileCategory = Common.tgwfile;
	Common.fileChanged = false;
	resetAllSetting();
	setTitle(translate("title.advancedmodenew"));
      }
    }

    // when Open is clicked
    if (e.getSource() == openFile) {
      if (Common.fileChanged) {		// current setting changed
	if (checkSave(2)) {			// prompt for save
	  addFileChooser();			// add a file chooser
	  int openReply = fileChooser.showOpenDialog(this);
	  if (openReply == JFileChooser.APPROVE_OPTION) {
	    Common.openedFileNum++;		// number of opened files
	    Common.fileState = Common.openedFile;
	    selectedFile = fileChooser.getSelectedFile(); // selected file
	    setWindowTitle(selectedFile.getAbsolutePath());
	    String fileExtension = Common.getFileExtension(selectedFile);
	    if (fileExtension.equalsIgnoreCase("tgd")) {
	      Common.fileCategory = Common.tgdfile;
	      // load TGD DOM tree from the selected file
	      loadTGDDOM(selectedFile.getAbsolutePath());
	      loadTGDsetting();			// set options in UI
	    }
	    else {
	      Common.fileCategory = Common.tgwfile;
	      // load TGD DOM tree from the selected file
	      loadTGWDOM(selectedFile.getAbsolutePath());
	      loadTGWsetting();			// set options in UI
	    }
	    storeSetting(loadedSetting);	// store settings to compare
	    Common.fileChanged = false;
	    changeRecentFilesDOM();		// change the DOM tree
	    Common.writeXML(domSetting, settingsFullPath); // output DOM tree
	  }
	}
      }
      else {
	addFileChooser();
	int openReply = fileChooser.showOpenDialog(this);
	if (openReply == JFileChooser.APPROVE_OPTION) {
	  Common.openedFileNum++;
	  Common.fileState = Common.openedFile;
	  selectedFile = fileChooser.getSelectedFile();
	  setWindowTitle(selectedFile.getAbsolutePath());
	  String fileExtension = Common.getFileExtension(selectedFile);
	  if (fileExtension.equalsIgnoreCase("tgd")) {
	    Common.fileCategory = Common.tgdfile;
	    // Load TGD DOM tree from the selected file
	    loadTGDDOM(selectedFile.getAbsolutePath());
	    loadTGDsetting();			// set options in UI
	  }
	  else {
	    Common.fileCategory = Common.tgwfile;
	    // Load TGD DOM tree from the selected file
	    loadTGWDOM(selectedFile.getAbsolutePath());
	    loadTGWsetting();			// set options in UI
	  }
	  storeSetting(loadedSetting);		// store settings to compare
	  Common.fileChanged = false;
	  changeRecentFilesDOM();		// change the DOM tree
	  Common.writeXML(domSetting, settingsFullPath); // output DOM tree
	}
      }
    }

    // when any Recent File is clicked
    if (e.getSource() == recentFile1 || e.getSource() == recentFile2 ||
	e.getSource() == recentFile3 || e.getSource() == recentFile4) {
      if (Common.fileChanged) {			// setting changed?
	if (checkSave(2)) {			// prompt for save?
	  Common.fileState = Common.openedFile;
	  String path = ((AbstractButton) e.getSource()).getText(); // file path
	  String extension = Common.getFileExtension(path);
	  setWindowTitle(path);
	  addFileChooser();
	  fileChooser.setSelectedFile(new File(path));
	  selectedFile = new File(path);
	  if (extension.equalsIgnoreCase("tgd")) {
	    Common.fileCategory = Common.tgdfile;
	    loadTGDDOM(path);
	    loadTGDsetting();
	  }
	  else if (extension.equalsIgnoreCase("tgw")) {
	    Common.fileCategory = Common.tgwfile;
	    loadTGWDOM(path);
	    loadTGWsetting();
	  }
	  storeSetting(loadedSetting);
	  Common.fileChanged = false;
	}
      }
      else {					// not changed
	Common.fileState = Common.openedFile;
	String path = ((AbstractButton) e.getSource()).getText(); // file path
	String extension = Common.getFileExtension(path);
	setWindowTitle(path);
	addFileChooser();
	fileChooser.setSelectedFile(new File(path));
	selectedFile = new File(path);
	if (extension.equalsIgnoreCase("tgd")) {
	  Common.fileCategory = Common.tgdfile;
	  loadTGDDOM(path);
	  loadTGDsetting();
	}
	else if (extension.equalsIgnoreCase("tgw")) {
	  Common.fileCategory = Common.tgwfile;
	  loadTGWDOM(path);
	  loadTGWsetting();
	}
	storeSetting(loadedSetting);
	Common.fileChanged = false;
      }
    }

    // when Save is clicked
    if (e.getSource() == save) {
      if (Common.fileState == Common.createdFile)
	saveAs.doClick();			// click SaveAs menu item
      else if (Common.fileState == Common.openedFile) {
	if (Common.fileCategory == Common.tgdfile) {
	  changeTGDDOM();
	  if (Common.getFileExtension(selectedFile).equalsIgnoreCase("tgd"))
	    warnSaveCompleted(domTGD, selectedFile, "none");
	  else
	    warnSaveCompleted(domTGD, selectedFile, "tgd");
	}
	else if (Common.fileCategory == Common.tgwfile) {
	  changeTGWDOM();
	  if (Common.getFileExtension(selectedFile).equalsIgnoreCase("tgw"))
	    warnSaveCompleted(domTGW, selectedFile, "none");
	  else
	    warnSaveCompleted(domTGW, selectedFile, "tgw");
	}
	else if (fileChooser.getFileFilter().getDescription().
		   equalsIgnoreCase("tgd")) {
	  Common.fileCategory = Common.tgdfile;
	  changeTGDDOM();
	  if (Common.getFileExtension(selectedFile).equalsIgnoreCase("tgd"))
	    warnSaveCompleted(domTGD, selectedFile, "none");
	  else
	    warnSaveCompleted(domTGD, selectedFile, "tgd");
	}
	else if (fileChooser.getFileFilter().getDescription().
		   equalsIgnoreCase("tgw")) {
	  Common.fileCategory = Common.tgwfile;
	  changeTGWDOM();
	  if (Common.getFileExtension(selectedFile).equalsIgnoreCase("tgw"))
	    warnSaveCompleted(domTGW, selectedFile, "none");
	  else
	    warnSaveCompleted(domTGW, selectedFile, "tgw");
	}
      }
      Common.fileChanged = false;
      storeSetting(loadedSetting);
    }

    // when Save As is clicked
    if (e.getSource() == saveAs) {
      addFileChooser();
      int saveReply = fileChooser.showSaveDialog(this);
      if (saveReply == JFileChooser.APPROVE_OPTION) {
	Common.fileState = Common.openedFile;
	selectedFile = fileChooser.getSelectedFile();
	Common.openedFileNum++;		// number of opened files
	String fileExtension = Common.getFileExtension(selectedFile);
	if (fileExtension.equalsIgnoreCase("tgd") ||
	    fileExtension.equalsIgnoreCase("tgw"))
	  setWindowTitle(selectedFile.getAbsolutePath());
	else {
	  fileExtension = fileChooser.getFileFilter().getDescription();
	  selectedFile =
	    new File(selectedFile.getAbsolutePath() + "." + fileExtension);
	  setWindowTitle(selectedFile.getAbsolutePath());
	}

	if (fileExtension.equalsIgnoreCase("tgd")) {
	  Common.fileCategory = Common.tgdfile;
	  changeTGDDOM();
	  warnSaveCompleted(domTGD, selectedFile, "none");
	}
	else if (fileExtension.equalsIgnoreCase("tgw")) {
	  Common.fileCategory = Common.tgwfile;
	  changeTGWDOM();
	  warnSaveCompleted(domTGW, selectedFile, "none");
	}
	else {
	  if (fileChooser.getFileFilter().getDescription().
		equalsIgnoreCase("tgd")) {
	    Common.fileCategory = Common.tgdfile;
	    changeTGDDOM();
	    warnSaveCompleted(domTGD, selectedFile, "tgd");
	  }
	  else if (fileChooser.getFileFilter().getDescription().
		     equalsIgnoreCase("tgw")) {
	    Common.fileCategory = Common.tgwfile;
	    changeTGWDOM();
	    warnSaveCompleted(domTGW, selectedFile, "tgw");
	  }
	}

	Common.fileChanged = false;
	changeRecentFilesDOM();		// change the DOM tree
	Common.writeXML(domSetting, settingsFullPath); // output DOM tree
	storeSetting(loadedSetting);
      }
    }

    // when Reload is clicked
    if (e.getSource() == reload) {
      if (Common.fileChanged) {
	if (checkSave(3)) {
	  if (Common.fileState == Common.openedFile) {
	    selectedFile = fileChooser.getSelectedFile();

	    String fileExtension = Common.getFileExtension(selectedFile);
	    if (fileExtension.equalsIgnoreCase("tgd")) {
	      Common.fileCategory = Common.tgdfile;
	      loadTGDDOM(selectedFile.getAbsolutePath());
	      loadTGDsetting();
	      setWindowTitle(selectedFile.getAbsolutePath());
	    }
	    else if (fileExtension.equalsIgnoreCase("tgw")) {
	      Common.fileCategory = Common.tgwfile;
	      loadTGWDOM(selectedFile.getAbsolutePath());
	      loadTGWsetting();
	      setWindowTitle(selectedFile.getAbsolutePath());
	    }
	    else {
	      if (Common.fileCategory == Common.tgdfile) {
		loadTGDDOM(selectedFile.getAbsolutePath() + ".tgd");
		loadTGDsetting();
		setWindowTitle(selectedFile.getAbsolutePath() + ".tgd");
	      }
	      if (Common.fileCategory == Common.tgwfile) {
		loadTGWDOM(selectedFile.getAbsolutePath() + ".tgw");
		loadTGWsetting();
		setWindowTitle(selectedFile.getAbsolutePath() + ".tgw");
	      }
	    }
	    Common.fileChanged = false;
	  }
	  else
	    JOptionPane.showMessageDialog(
	      this, translate("warn.reload.success"));
	}
      }
    }

    // when Exit is clicked
    if (e.getSource() == exit) {
      if (Common.fileChanged) {
	if (checkSave(0))
	  System.exit(0);
      }
      else
	System.exit(0);
    }

    // when Help is clicked
    if (e.getSource() == helpContent) {
      JDialog dialogue = new JDialog(this, "Wizard Help", false);
      Container container = dialogue.getContentPane();

      try {
	URL helpUrl =
	  getClass().getClassLoader().getResource("res/help.html");
	HTMLPane helpText = new HTMLPane(helpUrl);
	helpText.setEditable(false);
	JScrollPane helpScroll = new JScrollPane(helpText,
	  JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
	  JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
	helpScroll.setPreferredSize(new Dimension(
	  Common.helpWidth, Common.helpHeight));
	container.add(helpScroll);
	dialogue.pack();
	Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
	Dimension dialogueSize = dialogue.getPreferredSize();
	int dialogueX =
	  (int) ((screenSize.getWidth() - dialogueSize.getWidth()) / 2);
	int dialogueY =
	  (int) ((screenSize.getHeight() - dialogueSize.getHeight()) / 2);
	dialogue.setLocation(dialogueX, dialogueY);
	dialogue.setVisible(true);
	dialogue.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
      }
      catch (Exception exception) {
      }
    }

    // when Preference is clicked
    if (e.getSource() == preference)
      new SettingWindow(this, rootSetting);

    // when About is clicked
    if (e.getSource() == about)
      new AboutTerraWiz(this);

    if (Common.fileChanged) {
      if (!getTitle().endsWith("*"))
	setTitle(getTitle() + " *");

      reload.setEnabled(true);
    }
    else {
      if (getTitle().endsWith("*"))
	setTitle(getTitle().
	  substring(0, getTitle().length() - 2));
      reload.setEnabled(false);
    }
  }

  /**
    Save file with a warning.

    @param dom			document DOM
    @param selectedFile		selected file name
    @param suffix		selected file suffix
  */

  private void warnSaveCompleted(Document dom, File selectedFile,
   String suffix) {
    if (suffix.equals("tgd") || suffix.equals("tgw")) {
      if (!Common.writeXML(dom, selectedFile.getAbsolutePath() + "." + suffix))
	JOptionPane.showMessageDialog(this, translate("warn.savenotcomplete"));
      // else
	// JOptionPane.showMessageDialog(this, translate("warn.savecomplete"));

    }
    else if (suffix.equals("none")) {
      if (!Common.writeXML(dom, selectedFile.getAbsolutePath()))
	JOptionPane.showMessageDialog(this, translate("warn.savenotcomplete"));
      // else
	// JOptionPane.showMessageDialog(this, translate("warn.savecomplete"));
    }
  }

  /** Add a file chooser with two file extension to choose (tgd and tgw). */

  private void addFileChooser() {
    fileChooser = new JFileChooser();
    fileChooser.setCurrentDirectory(new File(homeDirectory));
    fileChooser.setFileFilter(new FileNameExtensionFilter("tgd", "tgd"));
    fileChooser.setFileFilter(new FileNameExtensionFilter("tgw", "tgw"));
  }

  /**
    Update recent files in DOM based on <var>selectedFile</var> with number
    <var>openedFileNum</var>.
  */

  private void changeRecentFilesDOM() {
    NodeList nodeList1 = rootSetting.getElementsByTagName("File");
    if (nodeList1.getLength() == 0 || nodeList1.getLength() < 4) {
      file = domSetting.createElement("File");
      fileTextNode = domSetting.createTextNode(selectedFile.getAbsolutePath());
      file.appendChild(fileTextNode);
      recentFile.appendChild(file);
    }
    else {
      recentFile.removeChild(nodeList1.item(0));
      file = domSetting.createElement("File");
      fileTextNode = domSetting.createTextNode(selectedFile.getAbsolutePath());
      file.appendChild(fileTextNode);
      recentFile.appendChild(file);
    }

    if (nodeList1.getLength() != 0) {
      int i = Common.openedFileNum % 4;
      switch (i) {
	  case 1:
	    recentFile1.setText(selectedFile.getAbsolutePath());
	    recentFile1.setVisible(true);
	    break;
	  case 2:
	    recentFile2.setText(selectedFile.getAbsolutePath());
	    recentFile2.setVisible(true);
	    break;
	  case 3:
	    recentFile3.setText(selectedFile.getAbsolutePath());
	    recentFile3.setVisible(true);
	    break;
	  case 0:
	    recentFile4.setText(selectedFile.getAbsolutePath());
	    recentFile4.setVisible(true);
	    break;
	  default:
	    break;
      }
    }
  }

  /**
    Check saved.

    @param category	kind of save
    @return 		true if to be saved
  */
  private boolean checkSave(int category) {
    int reply = 0;
    if (category == 1)				// new
      reply = JOptionPane.showConfirmDialog(this,
	translate("warn.newfilewithoutsave"), translate("warn.warning"), 2);

    else if (category == 0)			// exit
      reply = JOptionPane.showConfirmDialog(this,
	translate("warn.closefilewithoutsave"), translate("warn.warning"), 2);
    else if (category == 2)			// open
      reply = JOptionPane.showConfirmDialog(this,
	translate("warn.openfilewithoutsave"), translate("warn.warning"), 2);
    else if (category == 3)			// reload
      reply = JOptionPane.showConfirmDialog(this,
	translate("warn.reloadfilewithoutsave"), translate("warn.warning"), 2);

    return(reply == JFileChooser.APPROVE_OPTION);
  }

  /** Check all settings. */

  private void checkSetting() {
    for (int i = 0; i < tempSetting.length; i++)
      // both panels are disabled
      if (tempSetting[i][0] == 0 && loadedSetting[i][0] == 0)
	Common.fileChanged = false;
	// bth panels are enabled
      else if (tempSetting[i][0] == 1 && loadedSetting[i][0] == 1)
	// setting in a certain panel
	for (int j = 1; j < tempSetting[i].length; j++)
	  // all settings are same
	  if (tempSetting[i][j] == loadedSetting[i][j])
	    Common.fileChanged = false;
	  // different
	  else {
	    Common.fileChanged = true;
	    return;
	  }

      // one panel is enabled and another is disabled
      else {
	Common.fileChanged = true;
	return;
      }
  }

  /**
    Store the setting from the current setting to a two-dimensional array.
    The first element of each row represents "enabled" or not.

    @param setting	settings array
  */

  private void storeSetting(int[][] setting) {
    // lake
    if (cboxLake.isSelected()) {
      setting[0][0] = 1;
      if (radioLakeDepth1.isSelected())
	setting[0][1] = 0;
      else if (radioLakeDepth2.isSelected())
	setting[0][1] = 1;

      if (radioLakeRadius1.isSelected())
	setting[0][2] = 0;
      else if (radioLakeRadius2.isSelected())
	setting[0][2] = 1;

      if (radioLakeRough1.isSelected())
	setting[0][3] = 0;
      else if (radioLakeRadius2.isSelected())
	setting[0][3] = 1;

      if (radioLakeWave1.isSelected())
	setting[0][4] = 0;
      else if (radioLakeWave2.isSelected())
	setting[0][4] = 1;

      if (radioLakeReflect1.isSelected())
	setting[0][5] = 0;
      else if (radioLakeReflect2.isSelected())
	setting[0][5] = 1;
    }
    else
      setting[0][0] = 0;

    // mountain
    if (cboxMountain.isSelected()) {
      setting[1][0] = 1;
      if (radioMountRough1.isSelected())
	setting[1][1] = 0;
      else if (radioMountRough2.isSelected())
	setting[1][1] = 1;

      if (radioMountFlat1.isSelected())
	setting[1][2] = 0;
      else if (radioMountFlat2.isSelected())
	setting[1][2] = 1;

    }
    else
      setting[1][0] = 0;

    // surface
    if (cboxSurface.isSelected()) {
      setting[2][0] = 1;
      if (radioSurface1.isSelected())
	setting[2][1] = 0;
      else if (radioSurface2.isSelected())
	setting[2][1] = 1;

      if (radioSurfaceMaxalt1.isSelected())
	setting[2][2] = 0;
      else if (radioSurfaceMaxalt2.isSelected())
	setting[2][2] = 1;
      else if (radioSurfaceMaxalt3.isSelected())
	setting[2][2] = 2;

      if (radioSurfaceMinalt1.isSelected())
	setting[2][3] = 0;
      else if (radioSurfaceMinalt2.isSelected())
	setting[2][3] = 1;
      else if (radioSurfaceMinalt3.isSelected())
	setting[2][3] = 2;

    }
    else
      setting[2][0] = 0;

    // cloud
    if (cboxCloud.isSelected()) {
      setting[3][0] = 1;
      if (radioCloudCategory1.isSelected())
	setting[3][1] = 0;
      else if (radioCloudCategory2.isSelected())
	setting[3][1] = 1;

      if (radioCloudColor1.isSelected())
	setting[3][2] = 0;
      else if (radioCloudColor2.isSelected())
	setting[3][2] = 1;

      if (radioCloudAltitude1.isSelected())
	setting[3][3] = 0;
      else if (radioCloudAltitude2.isSelected())
	setting[3][3] = 1;

      if (radioCloudDepth1.isSelected())
	setting[3][4] = 0;
      else if (radioCloudDepth2.isSelected())
	setting[3][4] = 1;

      if (radioCloudDensity1.isSelected())
	setting[3][5] = 0;
      else if (radioCloudDensity2.isSelected())
	setting[3][5] = 1;

    }
    else
      setting[3][0] = 0;

    // lighting and camera panels always on
    setting[4][0] = 1;
    setting[5][0] = 1;
    if (radioLightHeading1.isSelected())
      setting[4][1] = 0;
    else if (radioLightHeading2.isSelected())
      setting[4][1] = 1;
    else if (radioLightHeading3.isSelected())
      setting[4][1] = 2;
    else if (radioLightHeading4.isSelected())
      setting[4][1] = 3;

    if (radioLightElevation1.isSelected())
      setting[4][2] = 0;
    else if (radioLightElevation2.isSelected())
      setting[4][2] = 1;
    else if (radioLightElevation3.isSelected())
      setting[4][2] = 2;

    if (radioCameraHeight1.isSelected())
      setting[5][1] = 0;
    else if (radioCameraHeight2.isSelected())
      setting[5][1] = 1;

    if (radioCameraHeading1.isSelected())
      setting[5][2] = 0;
    else if (radioCameraHeading2.isSelected())
      setting[5][2] = 1;
    else if (radioCameraHeading3.isSelected())
      setting[5][2] = 2;
    else if (radioCameraHeading4.isSelected())
      setting[5][2] = 3;
  }

  /** Check all the options in the interface. */

  private void checkOptions() {
    if (Common.fileState == Common.openedFile)
      reload.setEnabled(true);
    else
      reload.setEnabled(false);

    // lake checkbox
    if (cboxLake.isSelected())
      enableLake(true);
    else
      enableLake(false);

    // mountain checkbox
    if (cboxMountain.isSelected())
      enableMount(true);
    else
      enableMount(false);

    // surface checkbox
    if (cboxSurface.isSelected())
      enableSurface(true);
    else
      enableSurface(false);

    // cloud checkbox
    if (cboxCloud.isSelected())
      enableCloud(true);
    else
      enableCloud(false);
  }

  /** Change DOM of TGD. */

  private void changeTGDDOM() {
    // lake
    if (cboxLake.isSelected()) {
      if (Common.checkNodeExistNum(rootTGD, "lake") == 0)
	Common.enableTGDLakeNodes(rootTGD, domTGD, lakeTGD, waterShader);

      if (radioLakeDepth1.isSelected()) {		// shallow
	Common.changeAttr(rootTGD, "lake", "water_level",
	  Common.tgdLakeLevel1, 0);
	Common.changeAttr(rootTGD, "lake", "centre", "0 " +
	  Common.tgdLakeLevel1 + " 0", 0);
      }
      else if (radioLakeDepth2.isSelected()) {		// deep
	Common.changeAttr(rootTGD, "lake", "water_level",
	  Common.tgdLakeLevel2, 0);
	Common.changeAttr(rootTGD, "lake", "centre", "0 " +
	  Common.tgdLakeLevel2 + " 0", 0);
      }

      if (radioLakeRadius1.isSelected())		// narrow
	Common.changeAttr(rootTGD, "lake", "max_radius",
	  Common.tgdLakeRadius1, 0);
      else if (radioLakeRadius2.isSelected())		// broad
	Common.changeAttr(rootTGD, "lake", "max_radius",
	  Common.tgdLakeRadius2, 0);

      if (radioLakeRough1.isSelected())			// smooth
	Common.changeAttr(rootTGD, "water_shader", "roughness",
	  Common.tgdLakeRoughness1, 0);
      else if (radioLakeRough2.isSelected())		// Rough
	Common.changeAttr(rootTGD, "water_shader", "roughness",
	  Common.tgdLakeRoughness2, 0);

      if (radioLakeWave1.isSelected())			// small wave
	Common.changeAttr(rootTGD, "water_shader", "wave_scale",
	  Common.tgdLakeScale1, 0);
      else if (radioLakeWave2.isSelected())		// big wave
	Common.changeAttr(rootTGD, "water_shader", "wave_scale",
	  Common.tgdLakeScale2, 0);

      if (radioLakeReflect1.isSelected())		// low reflectivity
	Common.changeAttr(rootTGD, "water_shader", "master_reflectivity",
	  Common.tgdLakeReflec1, 0);
      else if (radioLakeReflect2.isSelected())		// high reflectivity
	Common.changeAttr(rootTGD, "water_shader", "master_reflectivity",
	  Common.tgdLakeReflec2, 0);

    }
    else
      Common.disableTGDLakeNodes(rootTGD);

    // mountain
    boolean isTGDMountExist = false;
    if (cboxMountain.isSelected()) {
      NodeList aList = rootTGD.getElementsByTagName("power_fractal_shader_v3");
      for (int i = 0; i < aList.getLength(); i++) {
	NamedNodeMap attributes = aList.item(i).getAttributes();
	for (int j = 0; j < attributes.getLength(); j++) {
	  Node theAttribute = attributes.item(j);
	  if (theAttribute.getNodeName().equals("gui_group") &&
	    theAttribute.getNodeValue().equals("Terrain"))
	    isTGDMountExist = true;

	}
      }
      if (!isTGDMountExist) {
	Common.enableTGDMountNodes(rootTGD, domTGD, fractalShader);
	Common.changeAttr(rootTGD, "compute_terrain", "input_node",
	  "Fractal terrain 01", 0);
      }
      if (radioMountRough1.isSelected())		// smooth
	Common.changeAttr(rootTGD, "power_fractal_shader_v3",
	  "feature_scale", Common.tgdMountainRoughness1, 1);
      else if (radioMountRough2.isSelected())		// rough
	Common.changeAttr(rootTGD, "power_fractal_shader_v3",
	  "feature_scale", Common.tgdMountainRoughness2, 1);

      if (radioMountFlat1.isSelected())			// flat
	Common.changeAttr(rootTGD, "power_fractal_shader_v3",
	  "smallest_scale", Common.tgdMountainFlat1, 1);
      else if (radioMountFlat2.isSelected())		// bumpy
	Common.changeAttr(rootTGD, "power_fractal_shader_v3",
	  "smallest_scale", Common.tgdMountainFlat2, 1);

    }
    else
      Common.disableTGDMountNodes(rootTGD);

    // surface
    if (cboxSurface.isSelected()) {
      if (Common.checkNodeExistNum(rootTGD, "surface_layer") == 0)
	Common.enableTGDSurfaceNodes(rootTGD, domTGD, surfaceLayer,
	  surfaceShader, cboxMountain);

      Common.changeAttr(rootTGD, "planet", "surface_shader",
	"Surface layer 01", 0);

      if (radioSurface1.isSelected())			// snow
	Common.changeAttr(rootTGD, "surface_layer", "diffuse_colour",
	  Common.tgdSurfaceSkin1, 0);

      else if (radioSurface2.isSelected())		// grass
	Common.changeAttr(rootTGD, "surface_layer", "diffuse_colour",
	  Common.tgdSurfaceSkin2, 0);

      if (radioSurfaceMaxalt1.isSelected()) {		// low max altitude
	Common.changeAttr(rootTGD, "surface_layer", "limit_maximum_altitude",
	  "1", 0);
	Common.changeAttr(rootTGD, "surface_layer", "maximum_altitude",
	  Common.tgdSurfaceMaxAlt1, 0);
      }
      else if (radioSurfaceMaxalt2.isSelected()) {	// high max altitude
	Common.changeAttr(rootTGD, "surface_layer", "limit_maximum_altitude",
	  "1", 0);
	Common.changeAttr(rootTGD, "surface_layer", "maximum_altitude",
	  Common.tgdSurfaceMaxAlt2, 0);
      }
      else if (radioSurfaceMaxalt3.isSelected())	// none
	Common.changeAttr(rootTGD, "surface_layer", "limit_maximum_altitude",
	  "0", 0);

      if (radioSurfaceMinalt1.isSelected()) {		// low min altitude
	Common.changeAttr(rootTGD, "surface_layer", "limit_minimum_altitude",
	  "1", 0);
	Common.changeAttr(rootTGD, "surface_layer", "minimum_altitude",
	  Common.tgdSurfaceMinAlt1, 0);
      }
      else if (radioSurfaceMinalt2.isSelected()) {	// high min altitude
	Common.changeAttr(rootTGD, "surface_layer", "limit_minimum_altitude",
	  "1", 0);
	Common.changeAttr(rootTGD, "surface_layer", "minimum_altitude",
	  Common.tgdSurfaceMinAlt2, 0);
      }
      else if (radioSurfaceMinalt3.isSelected())	// none
	Common.changeAttr(rootTGD, "surface_layer", "limit_minimum_altitude",
	  "0", 0);

    }
    else
      Common.disableTGDSurfaceNodes(rootTGD);

    // cloud
    if (cboxCloud.isSelected()) {
      if (Common.checkNodeExistNum(rootTGD, "cloud_layer_v2") == 0)
	Common.enableTGDCloudNodes(rootTGD, domTGD, cloudLayer,
	  cloudShader);

      if (radioCloudCategory1.isSelected()) {		// altocumulus
	Common.changeAttr(rootTGD, "planet", "atmosphere_shader",
	  Common.tgdCloudCategory1, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "name",
	  Common.tgdCloudCategory1, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "rendering_method",
	  Common.tgdCloudCategory1_3, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "quality",
	  Common.tgdCloudCategory1_4, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "number_of_samples",
	  Common.tgdCloudCategory1_5, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "feature_scale",
	  Common.tgdCloudCategory1_6, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "lead-in_scale",
	  Common.tgdCloudCategory1_7, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "noise_octaves",
	  Common.tgdCloudCategory1_8, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "roughness",
	  Common.tgdCloudCategory1_9, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "variation_method",
	  Common.tgdCloudCategory1_10, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "lead-in_warp_amount",
	  Common.tgdCloudCategory1_11, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2",
	  "less_warp_at_feature_scale", Common.tgdCloudCategory1_12, 0);
      }
      else if (radioCloudCategory2.isSelected()) {	// cirrus
	Common.changeAttr(rootTGD, "planet", "atmosphere_shader",
	  Common.tgdCloudCategory2, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "name",
	  Common.tgdCloudCategory2, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "rendering_method",
	  Common.tgdCloudCategory2_3, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "quality",
	  Common.tgdCloudCategory2_4, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "number_of_samples",
	  Common.tgdCloudCategory2_5, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "feature_scale",
	  Common.tgdCloudCategory2_6, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "lead-in_scale",
	  Common.tgdCloudCategory2_7, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "noise_octaves",
	  Common.tgdCloudCategory2_8, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "roughness",
	  Common.tgdCloudCategory2_9, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "variation_method",
	  Common.tgdCloudCategory2_10, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2", "lead-in_warp_amount",
	  Common.tgdCloudCategory2_11, 0);
	Common.changeAttr(rootTGD, "cloud_layer_v2",
	  "less_warp_at_feature_scale", Common.tgdCloudCategory2_12, 0);
      }

      if (radioCloudCategory2.isSelected())		// cirrus
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_colour",
	  Common.tgdCloudColor1, 0);
      else if (radioCloudColor2.isSelected())		// gray
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_colour",
	  Common.tgdCloudColor2, 0);

      if (radioCloudAltitude1.isSelected())		// low altitude
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_altitude",
	  Common.tgdCloudAlt1, 0);
      else if (radioCloudAltitude2.isSelected())	// high altitude
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_altitude",
	  Common.tgdCloudAlt2, 0);

      if (radioCloudDensity1.isSelected())		// low density
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_density",
	  Common.tgdCloudDensity1, 0);
      else if (radioCloudDensity2.isSelected())		// high density
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_density",
	  Common.tgdCloudDensity2, 0);

      if (radioCloudDepth1.isSelected())		// thin
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_depth",
	  Common.tgdCloudDepth1, 0);
      else if (radioCloudDepth2.isSelected())		// thick
	Common.changeAttr(rootTGD, "cloud_layer_v2", "cloud_depth",
	  Common.tgdCloudDepth2, 0);
    }
    else
      Common.disableTGDCloudNodes(rootTGD);

    // lighting
    if (radioLightHeading1.isSelected())		// heading 1
      Common.changeAttr(rootTGD, "sunlight", "heading",
	Common.tgdLightingHead1, 0);
    else if (radioLightHeading2.isSelected())		// heading 2
      Common.changeAttr(rootTGD, "sunlight", "heading",
	Common.tgdLightingHead2, 0);
    else if (radioLightHeading3.isSelected())		// heading 3
      Common.changeAttr(rootTGD, "sunlight", "heading",
	Common.tgdLightingHead3, 0);
    else if (radioLightHeading4.isSelected())		// heading 4
      Common.changeAttr(rootTGD, "sunlight", "heading",
	Common.tgdLightingHead4, 0);

    if (radioLightElevation1.isSelected())		// elevation 1
      Common.changeAttr(rootTGD, "sunlight", "elevation",
	Common.tgdLightingElevation1, 0);
    else if (radioLightElevation2.isSelected())		// elevation 2
      Common.changeAttr(rootTGD, "sunlight", "elevation",
	Common.tgdLightingElevation2, 0);
    else if (radioLightElevation3.isSelected())		// elevation 3
      Common.changeAttr(rootTGD, "sunlight", "elevation",
	Common.tgdLightingElevation3, 0);

    // camera
    if (radioCameraHeight1.isSelected())		// low view
      Common.changeAttr(rootTGD, "camera", "position",
	Common.tgdCameraView1, 0);
    else if (radioCameraHeight2.isSelected())		// high view
      Common.changeAttr(rootTGD, "camera", "position",
	Common.tgdCameraView2, 0);

    if (radioCameraHeading1.isSelected())		// east
      Common.changeAttr(rootTGD, "camera", "rotation",
	Common.tgdCameraHeading1, 0);
    else if (radioCameraHeading2.isSelected())		// north
      Common.changeAttr(rootTGD, "camera", "rotation",
	Common.tgdCameraHeading2, 0);
    else if (radioCameraHeading3.isSelected())		// west
      Common.changeAttr(rootTGD, "camera", "rotation",
	Common.tgdCameraHeading3, 0);
    else if (radioCameraHeading4.isSelected())		// south
      Common.changeAttr(rootTGD, "camera", "rotation",
	Common.tgdCameraHeading4, 0);
  }

  /** Change DOM of TGW. */

  private void changeTGWDOM() {
    // lake
    if (cboxLake.isSelected()) {
      if (Common.checkNodeExistNum(rootTGW, "lake") == 0)
	Common.enableTGWLakeNodes(rootTGW, domTGW, lakeTGW);

      if (radioLakeDepth1.isSelected())			// shallow
	Common.changeAttr(rootTGW, "lake", "water_level",
	  Common.tgwLakeLevel1, 0);
      else if (radioLakeDepth2.isSelected())		// deep
	Common.changeAttr(rootTGW, "lake", "water_level",
	  Common.tgwLakeLevel2, 0);

      if (radioLakeRadius1.isSelected())		// narrow
	Common.changeAttr(rootTGW, "lake", "max_radius",
	  Common.tgwLakeRadius1, 0);
      else if (radioLakeRadius2.isSelected())		// broad
	Common.changeAttr(rootTGW, "lake", "max_radius",
	  Common.tgwLakeRadius2, 0);

      if (radioLakeRough1.isSelected())			// smooth
	Common.changeAttr(rootTGW, "lake", "roughness",
	  Common.tgwLakeRoughness1, 0);
      else if (radioLakeRough2.isSelected())		// rough
	Common.changeAttr(rootTGW, "lake", "roughness",
	  Common.tgwLakeRoughness2, 0);

      if (radioLakeWave1.isSelected())			// small wave
	Common.changeAttr(rootTGW, "lake", "wave_scale",
	  Common.tgwLakeScale1, 0);
      else if (radioLakeWave2.isSelected())		// big wave
	Common.changeAttr(rootTGW, "lake", "wave_scale",
	  Common.tgwLakeScale2, 0);

      if (radioLakeReflect1.isSelected())		// low reflectivity
	Common.changeAttr(rootTGW, "lake", "reflectivity",
	  Common.tgwLakeReflect1, 0);
      else if (radioLakeReflect2.isSelected())		// high reflectivity
	Common.changeAttr(rootTGW, "lake", "reflectivity",
	  Common.tgwLakeReflect2, 0);
    }
    else
      Common.disableTGWLakeNodes(rootTGW);

    // mountain
    if (cboxMountain.isSelected()) {
      if (Common.checkNodeExistNum(rootTGW, "mountain") == 0)
	Common.enableTGWMountNodes(rootTGW, domTGW, mountain);

      if (radioMountRough1.isSelected())		// smooth
	Common.changeAttr(rootTGW, "mountain", "feature_scale",
	  Common.tgwMountainFeature1, 0);
      else if (radioMountRough2.isSelected())		// rough
	Common.changeAttr(rootTGW, "mountain", "feature_scale",
	  Common.tgwMountainFeature2, 0);

      if (radioMountFlat1.isSelected())			// flat
	Common.changeAttr(rootTGW, "mountain", "flat",
	  Common.tgwMountainFlat1, 0);
      else if (radioMountFlat2.isSelected())		// bumpy
	Common.changeAttr(rootTGW, "mountain", "flat",
	  Common.tgwMountainFlat2, 0);
    }
    else
      Common.disableTGWMountNodes(rootTGW);

    // surface
    if (cboxSurface.isSelected()) {
      if (Common.checkNodeExistNum(rootTGW, "surface") == 0)
	Common.enableTGWSurfaceNodes(rootTGW, domTGW, surface);

      if (radioSurface1.isSelected())			// snow
	Common.changeAttr(rootTGW, "surface", "skin",
	  Common.tgwSurfaceSkin1, 0);
      else if (radioSurface2.isSelected())		// grass
	Common.changeAttr(rootTGW, "surface", "skin",
	  Common.tgwSurfaceSkin2, 0);

      if (radioSurfaceMaxalt1.isSelected())		// snow
	Common.changeAttr(rootTGW, "surface", "maxAlt",
	  Common.tgwSurfaceMaxAlt1, 0);
      else if (radioSurfaceMaxalt2.isSelected())	// grass
	Common.changeAttr(rootTGW, "surface", "maxAlt",
	  Common.tgwSurfaceMaxAlt2, 0);
      else if (radioSurfaceMaxalt3.isSelected())	// grass
	Common.changeAttr(rootTGW, "surface", "maxAlt",
	  Common.tgwSurfaceMaxAlt3, 0);

      if (radioSurfaceMinalt1.isSelected())		// snow
	Common.changeAttr(rootTGW, "surface", "minAlt",
	  Common.tgwSurfaceMinAlt1, 0);
      else if (radioSurfaceMinalt2.isSelected())	// grass
	Common.changeAttr(rootTGW, "surface", "minAlt",
	  Common.tgwSurfaceMinAlt2, 0);
      else if (radioSurfaceMinalt3.isSelected())	// grass
	Common.changeAttr(rootTGW, "surface", "minAlt",
	  Common.tgwSurfaceMinAlt3, 0);
    }
    else
      Common.disableTGWSurfaceNodes(rootTGW);

    // cloud
    if (cboxCloud.isSelected()) {
      if (Common.checkNodeExistNum(rootTGW, "cloud") == 0)
	Common.enableTGWCloudNodes(rootTGW, domTGW, cloud);

      if (radioCloudCategory1.isSelected())		// altocumulus
	Common.changeAttr(rootTGW, "cloud", "category",
	  Common.tgwCloudCategory1, 0);
      else if (radioCloudCategory2.isSelected())	// cirrus
	Common.changeAttr(rootTGW, "cloud", "category",
	  Common.tgwCloudCategory2, 0);

      if (radioCloudColor1.isSelected())		// snow
	Common.changeAttr(rootTGW, "cloud", "colour",
	  Common.tgwCloudColour1, 0);
      else if (radioCloudColor2.isSelected())		// grass
	Common.changeAttr(rootTGW, "cloud", "colour",
	  Common.tgwCloudColour2, 0);

      if (radioCloudAltitude1.isSelected())		// snow
	Common.changeAttr(rootTGW, "cloud", "altitude",
	  Common.tgwCloudAlt1, 0);
      else if (radioCloudAltitude2.isSelected())	// grass
	Common.changeAttr(rootTGW, "cloud", "altitude",
	  Common.tgwCloudAlt2, 0);

      if (radioCloudDepth1.isSelected())		// snow
	Common.changeAttr(rootTGW, "cloud", "depth",
	  Common.tgwCloudDepth1, 0);
      else if (radioCloudDepth2.isSelected())		// grass
	Common.changeAttr(rootTGW, "cloud", "depth",
	  Common.tgwCloudDepth2, 0);

      if (radioCloudDensity1.isSelected())		// snow
	Common.changeAttr(rootTGW, "cloud", "density",
	  Common.tgwCloudDensity1, 0);
      else if (radioCloudDensity2.isSelected())		// grass
	Common.changeAttr(rootTGW, "cloud", "density",
	  Common.tgwCloudDensity2, 0);
    }
    else
      Common.disableTGWCloudNodes(rootTGW);

    if (radioLightHeading1.isSelected())
      Common.changeAttr(rootTGW, "lighting", "heading",
	Common.tgwLightingHeading1, 0);
    else if (radioLightHeading2.isSelected())
      Common.changeAttr(rootTGW, "lighting", "heading",
	Common.tgwLightingHeading2, 0);
    else if (radioLightHeading3.isSelected())
      Common.changeAttr(rootTGW, "lighting", "heading",
	Common.tgwLightingHeading3, 0);
    else if (radioLightHeading4.isSelected())
      Common.changeAttr(rootTGW, "lighting", "heading",
	Common.tgwLightingHeading4, 0);

    if (radioLightElevation1.isSelected())
      Common.changeAttr(rootTGW, "lighting", "elevation",
	Common.tgwLightingElevation1, 0);
    else if (radioLightElevation2.isSelected())
      Common.changeAttr(rootTGW, "lighting", "elevation",
	Common.tgwLightingElevation2, 0);
    else if (radioLightElevation3.isSelected())
      Common.changeAttr(rootTGW, "lighting", "elevation",
	Common.tgwLightingElevation3, 0);

    if (radioCameraHeight1.isSelected())
      Common.changeAttr(rootTGW, "camera", "height",
	Common.tgwCameraView1, 0);
    else if (radioCameraHeight2.isSelected())
      Common.changeAttr(rootTGW, "camera", "height",
	Common.tgwCameraView2, 0);

    if (radioCameraHeading1.isSelected())
      Common.changeAttr(rootTGW, "camera", "heading",
	Common.tgwCameraHeading1, 0);
    else if (radioCameraHeading2.isSelected())
      Common.changeAttr(rootTGW, "camera", "heading",
	Common.tgwCameraHeading2, 0);
    else if (radioCameraHeading3.isSelected())
      Common.changeAttr(rootTGW, "camera", "heading",
	Common.tgwCameraHeading3, 0);
    else if (radioCameraHeading4.isSelected())
      Common.changeAttr(rootTGW, "camera", "heading",
	Common.tgwCameraHeading4, 0);
  }

  /** Load TGD setting from DOM. */

  private void loadTGDsetting() {
    if (Common.checkNodeExistNum(rootTGD, "lake") > 0) {
      cboxLake.setSelected(true);
      checkOptions();
      if (Integer.parseInt(Common.getValue(
	    rootTGD, "lake", "water_level", 0)) <= Common.tgdLakeLevel1)
	radioLakeDepth1.setSelected(true);
      else
	radioLakeDepth2.setSelected(true);

      if (Integer.parseInt(Common.getValue(
	    rootTGD, "lake", "max_radius", 0)) <= Common.tgdLakeRadius1)
	radioLakeRadius1.setSelected(true);
      else
	radioLakeRadius2.setSelected(true);

      if (Double.parseDouble(Common.getValue(
	    rootTGD, "water_shader", "roughness", 0)) <=
	      Common.tgdLakeRoughness1)
	radioLakeRough1.setSelected(true);
      else
	radioLakeRough2.setSelected(true);

      if (Integer.parseInt(Common.getValue(
	    rootTGD, "water_shader", "wave_scale", 0)) <=
	      Common.tgdLakeScale1)
	radioLakeWave1.setSelected(true);
      else
	radioLakeWave2.setSelected(true);

      if (Double.parseDouble(Common.getValue(
	    rootTGD, "water_shader", "master_reflectivity", 0)) <=
	      Common.tgdLakeReflec1)
	radioLakeReflect1.setSelected(true);
      else
	radioLakeReflect2.setSelected(true);
    }
    else {
      cboxLake.setSelected(false);
      checkOptions();
    }
    int x = 0;
    // look for node power_fractal_shader_v3
    NodeList aList = rootTGD.getElementsByTagName("power_fractal_shader_v3");
    for (int i = 0; i < aList.getLength(); i++) {
      NamedNodeMap attributes = aList.item(i).getAttributes();
      for (int j = 0; j < attributes.getLength(); j++) {
	Node theAttribute = attributes.item(j);
	// distinguish according to gui_group
	if (theAttribute.getNodeName().equals("gui_group") &&
	  theAttribute.getNodeValue().equals("Terrain"))
	  x++;
      }
    }

    if (x == 1) {
      cboxMountain.setSelected(true);
      checkOptions();
      if (Integer.parseInt(Common.getValue(
	    rootTGD, "power_fractal_shader_v3", "feature_scale", 1)) <=
	      Common.tgdMountainRoughness2)
	radioMountRough2.setSelected(true);
      else
	radioMountRough1.setSelected(true);

      if (Double.parseDouble(Common.getValue(
	    rootTGD, "power_fractal_shader_v3", "smallest_scale", 1)) <=
	      Common.tgdMountainFlat1)
	radioMountFlat1.setSelected(true);
      else
	radioMountFlat2.setSelected(true);

    }
    else {
      cboxMountain.setSelected(false);
      checkOptions();
    }

    int y = 0;
    NodeList aList1 = rootTGD.getElementsByTagName("power_fractal_shader_v3");
    for (int i = 0; i < aList1.getLength(); i++) {
      NamedNodeMap attributes = aList1.item(i).getAttributes();
      for (int j = 0; j < attributes.getLength(); j++) {
	Node theAttribute = attributes.item(j);
	if (theAttribute.getNodeName().equals("gui_group") &&
	  theAttribute.getNodeValue().equals("Shaders"))
	  y++;
      }
    }
    if (y == 2) {
      cboxSurface.setSelected(true);
      checkOptions();
      if (Common.getValue(rootTGD, "surface_layer", "diffuse_colour",
	0).equals(Common.tgdSurfaceSkin1))
	radioSurface1.setSelected(true);
      else if (Common.getValue(rootTGD, "surface_layer", "diffuse_colour",
	0).equals(Common.tgdSurfaceSkin2))
	radioSurface2.setSelected(true);

      if (Common.getValue(rootTGD, "surface_layer", "limit_maximum_altitude",
	0).equals("0"))
	radioSurfaceMaxalt3.setSelected(true);
      else
	if (Integer.parseInt(Common.getValue(rootTGD, "surface_layer",
	  "maximum_altitude", 0)) <= Common.tgdSurfaceMaxAlt1)
	  radioSurfaceMaxalt1.setSelected(true);
	else
	  radioSurfaceMaxalt2.setSelected(true);

      if (Common.getValue(rootTGD, "surface_layer", "limit_minimum_altitude",
	0).equals("0"))
	radioSurfaceMinalt3.setSelected(true);
      else
	if (Integer.parseInt(Common.getValue(rootTGD, "surface_layer",
	  "minimum_altitude", 0)) <= Common.tgdSurfaceMinAlt1)
	  radioSurfaceMinalt1.setSelected(true);
	else
	  radioSurfaceMinalt2.setSelected(true);
    }
    else {
      cboxSurface.setSelected(false);
      checkOptions();
    }

    if (Common.checkNodeExistNum(rootTGD, "cloud_layer_v2") > 0) {
      cboxCloud.setSelected(true);
      checkOptions();

      if (Common.getValue(rootTGD, "cloud_layer_v2", "name",
	0).equals(Common.tgdCloudCategory1))
	radioCloudCategory1.setSelected(true);
      else if (Common.getValue(rootTGD, "cloud_layer_v2", "name",
	0).equals(Common.tgdCloudCategory2))
	radioCloudCategory2.setSelected(true);

      if (Common.getValue(rootTGD, "cloud_layer_v2", "cloud_colour",
	0).equals(Common.tgdCloudColor1))
	radioCloudColor1.setSelected(true);
      else if (Common.getValue(rootTGD, "cloud_layer_v2", "cloud_colour",
	0).equals(Common.tgdCloudColor2))
	radioCloudColor2.setSelected(true);

      if (Integer.parseInt(Common.getValue(rootTGD, "cloud_layer_v2",
	"cloud_altitude", 0)) <= Common.tgdCloudAlt1)
	radioCloudAltitude1.setSelected(true);
      else
	radioCloudAltitude2.setSelected(true);

      if (Double.parseDouble(Common.getValue(rootTGD, "cloud_layer_v2",
	"cloud_density", 0)) <= Common.tgdCloudDensity1)
	radioCloudDensity1.setSelected(true);
      else
	radioCloudDensity2.setSelected(true);

      if (Integer.parseInt(Common.getValue(rootTGD, "cloud_layer_v2",
	"cloud_depth", 0)) <= Common.tgdCloudDepth1)
	radioCloudDepth1.setSelected(true);
      else
	radioCloudDepth2.setSelected(true);
    }
    else {
      cboxCloud.setSelected(false);
      checkOptions();
    }

    int i =
      Integer.parseInt(Common.getValue(rootTGD, "sunlight", "heading", 0));
    if (i <= Common.tgdLightingHead1)
      radioLightHeading1.setSelected(true);
    else if (i > Common.tgdLightingHead1 && i <= Common.tgdLightingHead2)
      radioLightHeading2.setSelected(true);
    else if (i > Common.tgdLightingHead2 && i <= Common.tgdLightingHead3)
      radioLightHeading3.setSelected(true);
    else if (i > Common.tgdLightingHead3)
      radioLightHeading4.setSelected(true);

    int j =
      Integer.parseInt(Common.getValue(rootTGD, "sunlight", "elevation", 0));
    if (j <= Common.tgdLightingElevation1)
      radioLightElevation1.setSelected(true);
    else if (j > Common.tgdLightingElevation1 &&
	     j <= Common.tgdLightingElevation2)
      radioLightElevation2.setSelected(true);
    else if (j > Common.tgdLightingElevation2 &&
	     j <= Common.tgdLightingElevation3)
      radioLightElevation3.setSelected(true);


    if (Common.getValue(
	  rootTGD, "camera", "position", 0).equals(
	    Common.tgdCameraView1))
      radioCameraHeight1.setSelected(true);
    else if (Common.getValue(
	       rootTGD, "camera", "position", 0).equals(
		 Common.tgdCameraView2))
      radioCameraHeight2.setSelected(true);

    if (Common.getValue(rootTGD, "camera", "rotation", 0).equals(
	  Common.tgdCameraHeading1))
      radioCameraHeading1.setSelected(true);
    else if (Common.getValue(rootTGD, "camera", "rotation", 0).equals(
	       Common.tgdCameraHeading2))
      radioCameraHeading2.setSelected(true);
    else if (Common.getValue(rootTGD, "camera", "rotation", 0).equals(
	       Common.tgdCameraHeading3))
      radioCameraHeading3.setSelected(true);
    else if (Common.getValue(rootTGD, "camera", "rotation", 0).equals(
	       Common.tgdCameraHeading4))
      radioCameraHeading4.setSelected(true);
  }

  /** Load TGW setting from DOM. */

  private void loadTGWsetting() {

    if (Common.checkNodeExistNum(rootTGW, "lake") > 0) {
      cboxLake.setSelected(true);
      checkOptions();
      if (Common.getValue(rootTGW, "lake", "water_level",
	0).equalsIgnoreCase(Common.tgwLakeLevel1))
	radioLakeDepth1.setSelected(true);
      else
	radioLakeDepth2.setSelected(true);

      if (Common.getValue(rootTGW, "lake", "max_radius",
	0).equalsIgnoreCase(Common.tgwLakeRadius1))
	radioLakeRadius1.setSelected(true);
      else
	radioLakeRadius2.setSelected(true);

      if (Common.getValue(rootTGW, "lake", "roughness",
	0).equalsIgnoreCase(Common.tgwLakeRoughness1))
	radioLakeRough1.setSelected(true);
      else
	radioLakeRough2.setSelected(true);

      if (Common.getValue(rootTGW, "lake", "wave_scale",
	0).equalsIgnoreCase(Common.tgwLakeScale1))
	radioLakeWave1.setSelected(true);
      else
	radioLakeWave2.setSelected(true);

      if (Common.getValue(rootTGW, "lake", "reflectivity",
	0).equalsIgnoreCase(Common.tgwLakeReflect1))
	radioLakeReflect1.setSelected(true);
      else
	radioLakeReflect2.setSelected(true);
    }
    else {
      cboxLake.setSelected(false);
      checkOptions();
    }

    if (Common.checkNodeExistNum(rootTGW, "mountain") > 0) {
      cboxMountain.setSelected(true);
      checkOptions();
      if (Common.getValue(rootTGW, "mountain", "feature_scale",
	0).equalsIgnoreCase(Common.tgwMountainFeature1))
	radioMountRough1.setSelected(true);
      else
	radioMountRough2.setSelected(true);

      if (Common.getValue(rootTGW, "mountain", "flat",
	0).equalsIgnoreCase(Common.tgwMountainFlat1))
	radioMountFlat1.setSelected(true);
      else
	radioMountFlat2.setSelected(true);

    }
    else {
      cboxMountain.setSelected(false);
      checkOptions();
    }

    if (Common.checkNodeExistNum(rootTGW, "surface") > 0) {
      cboxSurface.setSelected(true);
      checkOptions();
      if (Common.getValue(rootTGW, "surface", "skin",
	    0).equalsIgnoreCase(Common.tgwSurfaceSkin1))
	radioSurface1.setSelected(true);
      else
	radioSurface2.setSelected(true);

      if (Common.getValue(rootTGW, "surface", "maxAlt",
	    0).equalsIgnoreCase(Common.tgwSurfaceMaxAlt1))
	radioSurfaceMaxalt1.setSelected(true);
      else if (Common.getValue(rootTGW, "surface", "maxAlt",
		 0).equalsIgnoreCase(Common.tgwSurfaceMaxAlt2))
	radioSurfaceMaxalt2.setSelected(true);
      else if (Common.getValue(rootTGW, "surface", "maxAlt",
		 0).equalsIgnoreCase(Common.tgwSurfaceMaxAlt3))
	radioSurfaceMaxalt3.setSelected(true);

      if (Common.getValue(rootTGW, "surface", "minAlt",
	    0).equalsIgnoreCase(Common.tgwSurfaceMinAlt1))
	radioSurfaceMinalt1.setSelected(true);
      else if (Common.getValue(rootTGW, "surface", "minAlt",
		 0).equalsIgnoreCase(Common.tgwSurfaceMinAlt2))
	radioSurfaceMinalt2.setSelected(true);
      else if (Common.getValue(rootTGW, "surface", "minAlt",
		 0).equalsIgnoreCase(Common.tgwSurfaceMinAlt3))
	radioSurfaceMinalt3.setSelected(true);

    }
    else {
      cboxSurface.setSelected(false);
      checkOptions();
    }

    if (Common.checkNodeExistNum(rootTGW, "cloud") > 0) {
      cboxCloud.setSelected(true);
      checkOptions();
      if (Common.getValue(rootTGW, "cloud", "category",
	    0).equalsIgnoreCase(Common.tgwCloudCategory1))
	radioCloudCategory1.setSelected(true);
      else
	radioCloudCategory2.setSelected(true);

      if (Common.getValue(rootTGW, "cloud", "colour",
	    0).equalsIgnoreCase(Common.tgwCloudColour1))
	radioCloudColor1.setSelected(true);
      else
	radioCloudColor2.setSelected(true);

      if (Common.getValue(rootTGW, "cloud", "altitude",
	    0).equalsIgnoreCase(Common.tgwCloudAlt1))
	radioCloudAltitude1.setSelected(true);
      else
	radioCloudAltitude2.setSelected(true);

      if (Common.getValue(rootTGW, "cloud", "depth",
	    0).equalsIgnoreCase(Common.tgwCloudDepth1))
	radioCloudDepth1.setSelected(true);
      else
	radioCloudDepth2.setSelected(true);

      if (Common.getValue(rootTGW, "cloud", "density",
	    0).equalsIgnoreCase(Common.tgwCloudDensity1))
	radioCloudDensity1.setSelected(true);
      else
	radioCloudDensity2.setSelected(true);
    }
    else {
      cboxCloud.setSelected(false);
      checkOptions();
    }

    String str = Common.getValue(rootTGW, "lighting", "heading", 0);
    if (str.equalsIgnoreCase(Common.tgwLightingHeading1))
      radioLightHeading1.setSelected(true);
    else if (str.equalsIgnoreCase(Common.tgwLightingHeading2))
      radioLightHeading2.setSelected(true);
    else if (str.equalsIgnoreCase(Common.tgwLightingHeading3))
      radioLightHeading3.setSelected(true);
    else if (str.equalsIgnoreCase(Common.tgwLightingHeading4))
      radioLightHeading4.setSelected(true);

    String str1 = Common.getValue(rootTGW, "lighting", "elevation", 0);
    if (str1.equalsIgnoreCase(Common.tgwLightingElevation1))
      radioLightElevation1.setSelected(true);
    else if (str1.equalsIgnoreCase(Common.tgwLightingElevation2))
      radioLightElevation2.setSelected(true);
    else if (str1.equalsIgnoreCase(Common.tgwLightingElevation3))
      radioLightElevation3.setSelected(true);

    if (Common.getValue(rootTGW, "camera", "height",
	  0).equalsIgnoreCase(Common.tgwCameraView1))
      radioCameraHeight1.setSelected(true);
    else
      radioCameraHeight2.setSelected(true);

    String str2 = Common.getValue(rootTGW, "camera", "heading", 0);
    if (str2.equalsIgnoreCase(Common.tgwCameraHeading1))
      radioCameraHeading1.setSelected(true);
    else if (str2.equalsIgnoreCase(Common.tgwCameraHeading2))
      radioCameraHeading2.setSelected(true);
    else if (str2.equalsIgnoreCase(Common.tgwCameraHeading3))
      radioCameraHeading3.setSelected(true);
    else if (str2.equalsIgnoreCase(Common.tgwCameraHeading4))
      radioCameraHeading4.setSelected(true);

  }

  /**
    Enable options in lake panel.

    @param enabled	whether enabled
  */

  private void enableLake(boolean enabled) {
    radioLakeDepth1.setEnabled(enabled);
    radioLakeDepth2.setEnabled(enabled);
    radioLakeRadius1.setEnabled(enabled);
    radioLakeRadius2.setEnabled(enabled);
    radioLakeRough1.setEnabled(enabled);
    radioLakeRough2.setEnabled(enabled);
    radioLakeWave1.setEnabled(enabled);
    radioLakeWave2.setEnabled(enabled);
    radioLakeReflect1.setEnabled(enabled);
    radioLakeReflect2.setEnabled(enabled);
  }

  /**
    Enable options in mountain panel.

    @param enabled	whether enabled
  */

  private void enableMount(boolean enabled) {
    radioMountRough1.setEnabled(enabled);
    radioMountRough2.setEnabled(enabled);
    radioMountFlat1.setEnabled(enabled);
    radioMountFlat2.setEnabled(enabled);
  }

  /**
    Enable options in surface panel.

    @param enabled	whether enabled
  */

  private void enableSurface(boolean enabled) {
    radioSurface1.setEnabled(enabled);
    radioSurface2.setEnabled(enabled);
    labSurfaceMax.setEnabled(enabled);
    labSurfaceMin.setEnabled(enabled);
    radioSurfaceMaxalt1.setEnabled(enabled);
    radioSurfaceMaxalt2.setEnabled(enabled);
    radioSurfaceMaxalt3.setEnabled(enabled);
    radioSurfaceMinalt1.setEnabled(enabled);
    radioSurfaceMinalt2.setEnabled(enabled);
    radioSurfaceMinalt3.setEnabled(enabled);
  }

  /**
    Enable options in cloud panel.

    @param enabled	whether enabled
  */

  private void enableCloud(boolean enabled) {
    radioCloudCategory1.setEnabled(enabled);
    radioCloudCategory2.setEnabled(enabled);
    radioCloudColor1.setEnabled(enabled);
    radioCloudColor2.setEnabled(enabled);
    radioCloudAltitude1.setEnabled(enabled);
    radioCloudAltitude2.setEnabled(enabled);
    radioCloudDepth1.setEnabled(enabled);
    radioCloudDepth2.setEnabled(enabled);
    radioCloudDensity1.setEnabled(enabled);
    radioCloudDensity2.setEnabled(enabled);
  }

  /** Reset all settings. */

  private void resetAllSetting() {
    resetLake();
    resetMount();
    resetSurface();
    resetCloud();
    resetLight();
    resetCamera();
    checkOptions();
  }

  /** Reset lake settings. */

  private void resetLake() {
    cboxLake.setSelected(false);
    radioLakeDepth1.setSelected(true);
    radioLakeRadius1.setSelected(true);
    radioLakeRough1.setSelected(true);
    radioLakeWave1.setSelected(true);
    radioLakeReflect1.setSelected(true);
  }

  /** Reset mountain settings. */

  private void resetMount() {
    cboxMountain.setSelected(false);
    radioMountRough1.setSelected(true);
    radioMountFlat1.setSelected(true);
  }

  /** Reset surface settings. */

  private void resetSurface() {
    cboxSurface.setSelected(false);
    radioSurface1.setSelected(true);
    radioSurfaceMaxalt1.setSelected(true);
    radioSurfaceMinalt1.setSelected(true);
  }

  /** Reset cloud settings. */

  private void resetCloud() {
    cboxCloud.setSelected(false);
    radioCloudCategory1.setSelected(true);
    radioCloudColor1.setSelected(true);
    radioCloudAltitude1.setSelected(true);
    radioCloudDensity1.setSelected(true);
    radioCloudDepth1.setSelected(true);
  }

  /** Reset lighting settings. */

  private void resetLight() {
    radioLightHeading1.setSelected(true);
    radioLightElevation1.setSelected(true);
  }

  /** Reset camera settings. */

  private void resetCamera() {
    radioCameraHeight1.setSelected(true);
    radioCameraHeading1.setSelected(true);
  }

  /**
    Show message in window title.

    @param message	message
  */

  private void setWindowTitle(String message) {
    setTitle(translate("title.advancedmode") + " " + message);
  }

}

