/******************************************************************
 *  (C) Copyright 1994; dit/upm
 *  Distributed under the conditions stated in the
 *  TOPO General Public License (see file LICENSE)
 ******************************************************************
 *  $Log$
 ******************************************************************/

/******************************************************************
 *  
 *  Santiago Pavon Gomez
 *
 *  18 April 1990
 *
 *  Substitution of formal variables with actual expressions.
 *
 *  COMPILATION FLAGS:
 *  
 *  TYPE DEFINITIONS:
 *    *SVdescriptorTyp : Descriptor of a substitution table SV
 *
 *  FUNCTION DEFINITIONS:
 *     CreateSV                      : create a SV
 *     FreeSV                        : free a SV
 *     IsEmptySV                     : say whether a SV is empty
 *     InsertSV                      : insert a pair [e/v] into a SV
 *     InsertBehDefSV                : insert in a SV all vars defined
 *                                     in a behaviour
 *     Insert_VAL_SV                 : insert in a SV all [e/v] of a VAL
 *     GetActExp_SV                  : look at the actual expression
 *                                     attached to a variable
 *     In_SV                         : say whether a variable is in a SV
 *     SubstExprSV                   : make a substitution SV in an expression
 *     SubstPredSV                   : make a substitution SV in a predicate
 *     Subst_OL_SV                   : make a substitution SV in an offer list
 *     Subst_EL_SV                   : make a substitution SV in
 *                                     an expression list
 *     Subst_VAL_RightSide_SV        : make a substitution SV in
 *                                     the right side of a VAL
 *     Subst_VAL_BothSides_SV        : make a substitution SV in
 *                                     the left side of a VAL
 *     Var_in_Expr_SV                : return TRUE if there is any
 *                                     variable of SV in an expression 
 *     Var_in_Pred_SV                : return TRUE if there is any
 *                                     variable of SV in a predicate
 *     Var_in_EL_SV                  : return TRUE if there is any
 *                                     variable of SV in an expression list
 *     Var_in_VAL_RightSide_SV       : return TRUE if there is any variable
 *                                     of SV in the right side of a VAL
 *     Var_in_VAL_BothSides_SV       : return TRUE if there is any
 *                                     variable of SV in a VAL
 *     Var_in_OL_SV                  : return TRUE if there is any
 *                                     variable of SV in an OL
 *     SubstBehSV                    : make a substitution SV in a behavior
 *     Make_VAL_SV                   : make the SV table denoted by a VAL
 *     GetFreeVarsBehSV              : free variables of a behaviour
 *     IsUsedInBehSV                 : say whether a SV can make any change
 *                                     in a behaviour
 *     
 *  LOG:
 *     10/07/92. santiago. 
 *     The implementation of the SV tables has been changed to BST.
 *     Note that now it is necessary to pass a pointer to the SV table
 *     to the functions.
 *
 ******************************************************************/


#ifndef basust_v_H
#define basust_v_H      


#include "baattr.h"

/* 
 * Descriptor of a substitution table SV.
 *
 */
typedef void * SVdescriptorTyp;


/* CreateSV
 * Create a SV table.
 */
extern SVdescriptorTyp  CreateSV();


/* FreeSV
 * Free a SV table.
 */
extern void FreeSV( /* td */ );
/* SVdescriptorTyp* td ; */


/* IsEmptySV
 * Return TRUE if the SV table is empty.
 */
extern boolean IsEmptySV( /* td */ );
/* SVdescriptorTyp td; */


/* InsertSV
 * Insert a formal variable v and an actual expression e in the table td.
 */
extern void InsertSV( /* v, e, td */ );
/* int        v;
   ExprTyp    e;
   SVdescriptorTyp*        td; */


/* InsertBehDefSV
 * Insert in the table td the variables defined in behaviour b, 
 * in order to redefine them.
 * Stop when plet or instantation are reached.  
 */
extern void InsertBehDefSV( /* b, td */ );
/*  BehTyp   b;
    SVdescriptorTyp* td ;    */


/* Insert_VAL_SV
 * Insert in the table a variable assignment List.
 * Ignore void assignments ( x:s=x ).
 */
extern void Insert_VAL_SV( /* val, td */ );
/*  PVarAssignListTyp  val;
    SVdescriptorTyp*   td;   */


/* GetActExp_SV
 * Return the actual expression associated to the formal variable v.
 * It must be copied.
 */
extern ExprTyp GetActExp_SV( /* v, td */ );
/* int   v;
   SVdescriptorTyp*   td; */


/* In_SV
 * True if the variable v is in the table td.
 */
extern boolean In_SV( /* v, td */ );
/*   int              v;
     SVdescriptorTyp* td ; */


/* SubstExprSV
 * Make the replacement indicated in the table td in the expression 
 * pointed by pe.
 * Return true if any replacement has been actually made.
 */
extern boolean SubstExprSV( /* pe, td */ );
/* ExprTyp  * pe;
   SVdescriptorTyp*        td ;  */


/* SubstPredSV
 * Make the replacement indicated in the table td in the predicate p.
 * Return true if any replacement has been made actually.
 */
extern boolean SubstPredSV( /* pp, td */ );
/* PredicateTyp    *pp;
   SVdescriptorTyp *td ;  */


/* Subst_OL_SV
 * Make the replacement indicated in the table td in the offer list ol.
 */
extern void Subst_OL_SV( /* ol, td */ );
/*  OfferListTyp     ol;
    SVdescriptorTyp* td;  */


/* Subst_EL_SV
 * Make the replacement indicated in the table td in the expression list el.
 */
extern void Subst_EL_SV( /* el, td */ );
/* ExprListTyp      el;
   SVdescriptorTyp* td;  */


/* Subst_VAL_RightSide_SV
 * Make the replacement indicated by the table td
 * i the right side of the var assign list val.
 */
extern void Subst_VAL_RightSide_SV( /* val, td */ );
/*     VarAssignListTyp  val;
       SVdescriptorTyp*  td; */


/* Subst_VAL_BothSides_SV
 * Make the replacement indicated by the table td
 * in both sides of the var assign list val.
 */
extern void Subst_VAL_BothSides_SV( /* val, td */ );
/*     VarAssignListTyp val;
       SVdescriptorTyp* td; */


/* Var_in_Expr_SV
 * Return true if the expression e contains any formal variable that is
 * in the table td.
 */
extern boolean Var_in_Expr_SV( /* e, td */ );
/* ExprTyp   e;
   SVdescriptorTyp*       td; */


/* Var_in_Pred_SV
 * Return true if the predicate p contains any formal variable that is
 * in the table td.
 */
extern boolean Var_in_Pred_SV( /* p, td */ );
/* PredicateTyp      p;
   SVdescriptorTyp *td; */


/* Var_in_EL_SV
 * Return true if the expression list el contains any formal variable 
 * that is in the table td.
 */
extern boolean Var_in_EL_SV( /* el, td */ );
/* ExprListTyp      el;
   SVdescriptorTyp* td; */


/* Var_in_VAL_RightSide_SV
 * Return true if the right hand side of the var assign list val
 * contains any formal variable that is in the table td.
 */
extern boolean Var_in_VAL_RightSide_SV( /* val, td */ );
/*     VarAssignListTyp val;
       SVdescriptorTyp* td; */


/* Var_in_VAL_BothSides_SV
 * Return true if the var assign list val contains any formal variable 
 * that is in the table td.
 */
extern boolean Var_in_VAL_BothSides_SV( /* val,td */ );
/*     VarAssignListTyp val;
       SVdescriptorTyp* td; */


/* Var_in_OL_SV
 * Return true if the offer list ol contains any formal variable 
 * that is in the table td.
 */
extern boolean Var_in_OL_SV( /* ol,td */ );
/* OfferListTyp      ol;
   SVdescriptorTyp*  td; */


/* SubstBehSV
 * Make the replacement indicated in the table td in all the 
 * expressions of the behaviour b.
 */
extern void SubstBehSV( /* b, td */ );
/*  BehTyp           b;
    SVdescriptorTyp* td;  */


/* Make_VAL_SV
 * Make an attribute VAL from the table td.
 */
extern VarAssignListTyp Make_VAL_SV( /* td */ );
/* SVdescriptorTyp* td ; */


/* GetFreeVarsBeh
 * Return a list of free variables in b, including their parameterized value.
 *  ( variables undefined in b ).
 */
extern ExprListTyp GetFreeVarsBehSV( /* b */ );
/*     BehTyp b; */


/* IsUsedInBehSV
 * Say if a VAL (in sv-format) is actually used in a behaviour b.
 * (i.e. b contains variables defined in VAL )
 */
extern boolean IsUsedInBehSV( /* td, b */ );
/* SVdescriptorTyp *td ;
   BehTyp           b; */


/* SubstExprNoRwSV
 * Make the replacement indicated in the table td in the expression 
 * pointed by pe without rewriting.
 * Return TRUE if any replacement has been made.
 */  
extern boolean SubstExprNoRwSV ( /* pe, td */ );
/* ExprTyp         *pe;
   SVdescriptorTyp *td ; */


#endif


