/***********************************
  (C) Copyright 1992-1993; dit/upm
  Distributed under the conditions stated in the
  TOPO General Public License (see file LICENSE)
  ***********************************/

/***********************************
  
  David Larrabeiti Lopez
  
  Feb 15, 1991
  
  Management of one-immediate behaviours IB.
  
  Mxk
  b= ( SUMk( CHk Gk gk;Bk ) )
  1
  where:
  
  gk:= visible-action | i-action | exit        (Mxk>1)
  gk:= visible-action | i-action | exit | stop (Mxk=1)
  
  
  Let us define UB (unitary behaviour) as a chain of cells
  with one operand utmost.
  
  Example: any branch of a one-immediate behaviour down to
  its action : Chi Gi gi
  
  Ch1
  Ch2
  Gd1
  action
  
  COMPILATION OPTIONS: The behaviour of this module can be modified
  by the following compilation flags:
  (none)
  
  
  TYPE DEFINITIONS:
  (none)
  
  FUNCTION DEFINITIONS:
  AppendUB : append a unitary behaviour to a one-immediate behaviour
  GetArgIB : get one of the choices of a one-immediate behaviour
  LookArgIB : look at one of the choices of a one-immediate behaviour
  NumArgIB : number of choices of a one-immediate behaviour
  ApplyIB : apply a funtion to all branches of a one-immediate behaviour
  LookActionArgIB : look at the first action found in a branch of an ib
  LookLastUB : look at the last cell of a unitary behaviour
  LookActionUB : look at the first action found in a unitary behaviour
  LookPreActionUB : look at the father of the first action of a ub
  LookPreLastUB : look at the father of the last action of a ub
  SearchCellUntilActUB : look for a determined type cell in a ub
  CopyEqualCellsUB : copy all consecutive cells of the same type from
  a unitary behaviour
  ConcatUB : concatenate two unitary behaviours
  ExtractUB : extract all cells of a determined type from 2 ub's
  CountUB : number of cells composing a ub
  RemoveCellUB : remove a cell from a ub . 
  
  UPDATES:
  10-7-1992. David.
  New function to remove a cell from a unary behaviour.
  
  ************************************/


#ifndef EXimmed_H
#define EXimmed_H


#include "babeh.h"
#include "baattr.h"


/******************************************************************
 *                                                                                                *
 *              Unitary Behaviour (UB) functions                          *
 *                                                                *
 *      UB is supposed to be a behaviour composed of              *
 *      one-operand cells                                         *
 *                                                                                                *
 *******************************************************************/

/* LookLastUB
 * Last cell in a UB. ( NumArg(LookLastUB(b))==0 ).
 */
extern BehTyp LookLastUB(/*b*/);
/* BehTyp b; */

/*----------------------------------------------------------------*/

/* LookActionUB
 * First Action found in a UB .
 * ( Action:= [ gate | i | stop | exit ] ).
 */
extern BehTyp LookActionUB(/*b*/);
/* BehTyp b;*/

/*----------------------------------------------------------------*/

/* LookPreActionUB
 * Cell just before the Action in a UB .
 * ( Action:= [ gate | i | stop | exit ] ).
 * If there is not such a cell returns NULL.
 */
extern BehTyp LookPreActionUB(/*b*/);
/*     BehTyp b;*/

/*----------------------------------------------------------------*/

/* LookPreLastUB
 * Cell just before the last cell in a UB .
 * If there is not such a cell returns NULL.
 */
extern BehTyp LookPreLastUB(/*b*/);
/*     BehTyp b; */

/*----------------------------------------------------------------*/

/* SearchCellUntilActUB
 * Returns the first cell of type t that is found.
 * NULL if either an action or the end of the UB is found
 * without having found the cell.
 */
extern BehTyp SearchCellUntilActUB(/* b, t */);
/*     BehTyp b;
       CellTypeTyp t;*/

/*----------------------------------------------------------------*/

/* CopyEqualCellsUB
 * Returns a copy of the piece of UB whose cell types are equal to that given.
 * Example:
 *           Ch1<-     case b=Ch1:
 *           Ch2                   return a copy of Ch1
 *           Gd1<-                                  Ch2
 *           Gd2       case b=Gd1:
 *           Gd3                   return a copy of Gd1
 *           gate                                   Gd2
 *            :                                     Gd3
 *            :
 *            :
 */
extern BehTyp CopyEqualCellsUB(/*b*/);
/*     BehTyp b;*/

/*----------------------------------------------------------------*/

/* ConcatUB
 * Joins Two UBs . b2 becomes argument of LookLastUB(b1).
 * Any of them can be NULL.
 */
extern BehTyp ConcatUB(/*b1,b2*/);
/*     BehTyp b1,b2;*/

/*----------------------------------------------------------------*/

/* ExtractUB
 * Picks up (copies) the cells of type t from b1 and b2 and concatenates
 * them toLookher through ConcatUB.
 * Example:
 *           b1->Ch1     b2->Ch3  ExtractUB(b1, b2, GuardC)== copy of Gd1
 *               Ch2         Gd3                                      Gd2
 *               Gd1         Gd4                                      Gd3
 *               Gd2         Gd5                                      Gd4
 *               action      stop                                     Gd5
 *                : 
 *                :               ExtractUB(b1, b2,ChoiceC)== copy of Ch1
 *                                                                    Ch2
 *                                                                    Ch3
 */
extern BehTyp ExtractUB(/* b1, b2, t */);
/*     BehTyp b1,b2;
       CellTypeTyp t;*/

/*----------------------------------------------------------------*/

/* CountUB
 * Number of cells of a UB b
 */
extern int CountUB(/* b */);
/*     BehTyp b; */

/*----------------------------------------------------------------*/

/* RemoveCellUB
 * Remove cell n from UB b. 
 */
extern BehTyp RemoveCellUB(/* victim, b */);
/*       BehTyp victim,b; */


/******************************************************************
 *                                                                                                *
 *              One-immediate Behaviour (IB) functions                    *
 *                                                                *
 *      IB is supposed to be a choice of UBs                      *
 *                                                                                                *
 *******************************************************************/

/* AppendUB
 * Appends argument b to a one-immediate behaviour ib.
 * ib or b can be NULL.
 * if ib has one operand then ib and b get joined by an alternative
 */
extern BehTyp AppendUB( /* altB, b */ );
/* BehTyp altB,b;  */

/*----------------------------------------------------------------*/

/* SubtractUB
 * Gets the n-th argument of a one-immediate behaviour *ib.
 * if *ib has only one operand and n=1 then *ib becomes NULL
 * SubtractUB differs from GetArgIB in the treatment of alternative cells:
 * SubtractUB removes the alternative cell where the n-th operand is placed.
 */
extern BehTyp SubtractUB(/* ib, n */);
/*     BehTyp *ib;
       int     n; */

/*----------------------------------------------------------------*/

/* GetArgIB
 * Gets argument i from a one-immediate behaviour b for modification purposes.
 * This alternative will be reinserted afterwards.
 */
extern BehTyp GetArgIB( /* b, i */ );
/* BehTyp b;
   int i;    */

/*----------------------------------------------------------------*/

/* LookArgIB
 * Looks argument i from a one-immediate behaviour b.
 */
extern BehTyp LookArgIB(/* b, i */);
/* BehTyp b;
   int    i;  */

/*----------------------------------------------------------------*/

/* NumArgIB
 * Returns the number of arguments in a one-immediate behaviour.
 */
extern int NumArgIB(/* b */);
/* BehTyp b;  */

/*----------------------------------------------------------------*/

/* ApplyFuncIB
 * Applies function f to every UB of a one-immediate behaviour b.
 */
extern BehTyp ApplyFuncIB(/* b, f */);
/*     BehTyp b;
       BehTyp (*f)(); */

/*----------------------------------------------------------------*/

/* LookActionArgIB
 * Look at the action of the i-branch of a one-immediate behaviour b.
 */
extern BehTyp LookActionArgIB(/* b, i */);
/*     BehTyp b;
       int i; */

/*----------------------------------------------------------------*/

/* ExtractIB
 * return the alternatives whose action is of type t
 * *b will contain the rest
 * ExtractIB deals with NULL values.
 */
extern BehTyp ExtractIB(/* b, t */);
/*       BehTyp     *b;
	 CellTypeTyp t; */

/*----------------------------------------------------------------*/




#endif
